;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns iso.core
  (:require [utilis.map :refer [map-keys map-vals]]
            [utilis.types.string :refer [->string]]
            [jsonista.core :as json]
            [inflections.core :refer [hyphenate]]
            [clojure.java.io :as io]
            [clojure.string :as st]))

(declare iso-639-alpha-2-map iso-639-alpha-3-map
         iso-3166-alpha-2-map iso-3166-alpha-3-map)

(defn iso-639
  [code]
  (let [normalized-code (st/lower-case (->string code))]
    (case (count normalized-code)
      2 (get iso-639-alpha-2-map normalized-code)
      3 (get iso-639-alpha-3-map normalized-code)
      (throw (ex-info "Invalid language code length. Must be 2 or 3 characters" {:code code})))))

(defn iso-639-alpha-2-languages
  []
  (keys iso-639-alpha-2-map))

(defn iso-639-alpha-3-languages
  []
  (keys iso-639-alpha-3-map))

(defn iso-3166
  [code]
  (let [normalized-code (st/lower-case (->string code))]
    (case (count normalized-code)
      2 (get iso-3166-alpha-2-map normalized-code)
      3 (get iso-3166-alpha-3-map normalized-code)
      (throw (ex-info "Invalid country code length. Must be 2 or 3 characters" {:code code})))))

(defn iso-3166-alpha-2-country-codes
  []
  (keys iso-3166-alpha-2-map))

(defn iso-3166-alpha-3-country-codes
  []
  (keys iso-3166-alpha-3-map))

;;; Private

(defonce ^:private iso-639-alpha-2-map
  (->> (slurp (io/resource "standards/iso-639-1.json"))
       json/read-value
       (map-vals (fn [lang]
                   (->> lang
                        (map (fn [[key value]]
                               [(-> (case key
                                      "639-1" "alpha-2"
                                      "639-2" "alpha-3"
                                      "639-2/B" "alpha-3-alt"
                                      key)
                                    hyphenate
                                    keyword) value]))
                        (into {}))))))

(defonce ^:private iso-639-alpha-3-map
  (->> iso-639-alpha-2-map vals
       (mapcat (fn [lang]
                 [[(get lang :alpha-3) lang]
                  (when-let [alpha-3-alt (get lang :alpha-3-alt)]
                    [alpha-3-alt lang])]))
       (into {})))

(defonce ^:private iso-3166-list
  (->> (slurp (io/resource "standards/iso-3166.json"))
       json/read-value
       (map #(map-keys keyword
                       (-> (dissoc % "iso_3166-2")
                           (update "alpha-2" st/lower-case)
                           (update "alpha-3" st/lower-case))))))

(defonce ^:private iso-3166-alpha-2-map
  (reduce (fn [m country]
            (assoc m (:alpha-2 country) country))  {} iso-3166-list))

(defonce ^:private iso-3166-alpha-3-map
  (reduce (fn [m country]
            (assoc m (:alpha-3 country) country))  {} iso-3166-list))
