(ns formal.form.validation
  (:require [helix.core :refer [<>]]
            [helix.hooks :refer [use-effect use-ref]]
            [signum.subs :refer [reg-sub subscribe]]
            [ventus.hooks :refer [use-signal]]
            [ventus.macros :refer [defnc]]
            [utilis.js :as j]
            [utilis.fn :refer [fsafe]]))

(reg-sub
 :formal.form.validation/deref-all
 (fn [[_ signals]]
   (mapv deref signals)))

(defn registry-values
  ([registry] (registry-values registry nil))
  ([registry {:keys [subscribe-signals?]
              :or {subscribe-signals? true}}]
   (let [input-signals (->> registry :inputs vals)
         inputs (if subscribe-signals?
                  (use-signal (subscribe [:formal.form.validation/deref-all input-signals]))
                  (mapv deref input-signals))
         error? (boolean
                 (when (seq inputs)
                   (some :error inputs)))
         has-all-inputs? (->> inputs
                              (some (fn [{:keys [required raw-value error]}]
                                      (or error
                                          (and required
                                               (not (if (or (coll? raw-value)
                                                            (string? raw-value))
                                                      (seq raw-value)
                                                      (some? raw-value)))))))
                              not)]
     (merge {:inputs inputs
             :error? error?
             :has-all-inputs? has-all-inputs?}
            (when (some (comp some? :raw-value) inputs)
              {:value inputs})))))

(defn for-each-submit-button
  [form-ref f]
  (when-let [form @form-ref]
    (let [buttons (j/call form :getElementsByTagName "button")]
      (doseq [index (range (j/get buttons :length))]
        (let [button (j/get buttons index)]
          (when (= "submit" (j/call button :getAttribute "type"))
            (f button)))))))

(defn toggle-buttons-disabled
  [_registry form-ref disabled?]
  (use-effect
   [disabled?]
   (for-each-submit-button
    form-ref (fn [button]
               (if disabled?
                 (j/call button :setAttribute "disabled" "")
                 (j/call button :removeAttribute "disabled"))))
   nil))

(defnc watcher
  [{:keys [registry on-error on-value on-incomplete form-ref on-value on-change]}]
  (let [registry (use-signal registry)
        {:keys [inputs error? has-all-inputs?]} (registry-values registry)]
    (toggle-buttons-disabled registry form-ref (or error? (not has-all-inputs?)))
    (use-effect
     [inputs]
     (let [has-some-input? (boolean
                            (some (fn [{:keys [initial raw-value]}]
                                    (and (some? raw-value)
                                         (not initial)))
                                  inputs))]
       ((fsafe on-change) inputs)
       (cond
         error? ((fsafe on-error) inputs)

         (and has-some-input? has-all-inputs?)
         ((fsafe on-value) inputs)

         (and has-some-input? (not has-all-inputs?))
         ((fsafe on-incomplete) inputs)

         :else nil)))
    (<>)))
