(ns formal.form.validation
  (:require [helix.core :refer [$ <> defnc]]
            [helix.dom :refer [div] :as dom]
            [helix.children :as hch]
            [signum.signal :refer [signal alter!]]
            [signum.hooks :refer [use-signal]]
            [signum.subs :refer [reg-sub subscribe]]
            [react :as react]
            [utilis.js :as j]
            [utilis.fn :refer [fsafe]]))

(reg-sub
 :formal.form.validation/deref-all
 (fn [[_ signals]]
   (mapv deref signals)))

(defn registry-values
  ([registry] (registry-values registry nil))
  ([registry {:keys [subscribe-signals?]
              :or {subscribe-signals? true}}]
   (let [input-signals (->> registry :inputs vals)
         inputs (if subscribe-signals?
                  (use-signal (subscribe [:formal.form.validation/deref-all input-signals]))
                  (mapv deref input-signals))
         error? (boolean
                 (when (seq inputs)
                   (some :error inputs)))
         has-all-inputs? (->> inputs
                              (some (fn [{:keys [required raw-value error]}]
                                      (or error
                                          (and required
                                               (not (if (or (coll? raw-value)
                                                            (string? raw-value))
                                                      (seq raw-value)
                                                      (some? raw-value)))))))
                              not)]
     (merge {:inputs inputs
             :error? error?
             :has-all-inputs? has-all-inputs?}
            (when (some (comp some? :raw-value) inputs)
              {:value inputs})))))

(defn for-each-submit-button
  [ref f]
  (when-let [node (j/get ref :current)]
    (when-let [form (j/call node :closest "form")]
      (let [buttons (j/call form :getElementsByTagName "button")]
        (doseq [index (range (j/get buttons :length))]
          (let [button (j/get buttons index)]
            (when (= "submit" (j/call button :getAttribute "type"))
              (f button))))))))

(defn toggle-buttons-disabled
  [registry ref disabled?]
  (react/useEffect
   (fn []
     (for-each-submit-button
      ref (fn [button]
            (if disabled?
              (j/call button :setAttribute "disabled" "")
              (j/call button :removeAttribute "disabled"))))
     (fn []))
   #js [disabled?]))

(defnc watcher
  [{:keys [registry on-error on-value on-incomplete on-value on-change]}]
  (let [ref (react/useRef nil)
        registry (use-signal registry)
        {:keys [inputs value error? has-all-inputs?]} (registry-values registry)]
    (toggle-buttons-disabled registry ref (or error? (not has-all-inputs?)))
    (react/useEffect
     (fn []
       (let [has-some-input? (boolean
                              (some (fn [{:keys [initial raw-value]}]
                                      (and (some? raw-value)
                                           (not initial)))
                                    inputs))]
         ((fsafe on-change) inputs)
         (cond
           error? ((fsafe on-error) inputs)

           (and has-some-input? has-all-inputs?)
           ((fsafe on-value) inputs)

           (and has-some-input? (not has-all-inputs?))
           ((fsafe on-incomplete) inputs)

           :else nil))
       (fn []))
     #js [inputs])
    (div {:ref ref})))
