;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fluxus.flow
  (:refer-clojure :exclude [error-handler])
  (:require [fluxus.promise :as p]
            #?(:clj [fluxus.fiber :refer [fiber]])
            [utilis.map :refer [map-vals]]
            #?(:clj  [clojure.core.async :as a]
               :cljs [cljs.core.async :as a])
            [utilis.string :as ust]
            #?(:cljs [cljs.core.async.impl.buffers :refer
                      [RingBuffer FixedBuffer SlidingBuffer DroppingBuffer]])
            #?(:clj [clojure.pprint :refer [simple-dispatch]]))
  #?(:clj (:import [clojure.core.async.impl.buffers
                    FixedBuffer SlidingBuffer DroppingBuffer]
                   [clojure.lang IObj IMeta]
                   [java.util LinkedList])))

#?(:cljs (set! *warn-on-infer* true))

(defprotocol IFlow
  (close!  [flow] [flow {:keys [drain] :or {drain true}}])
  (closed? [flow])

  (on-close [flow f])
  (on-error [flow f])

  (put!  [flow v])
  (take! [flow] [s default]))

(declare pr-flow create-ch)

(deftype Flow [label read write closed handlers pending meta-map]
  Object
  (toString
    [^Flow this]
    (pr-flow this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash)
    [_]
    (hash [:fluxus/flow read]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv)
    [^Flow this other]
    (boolean
     (when (instance? Flow other)
       (and (= (#?(:clj .read :cljs .-read) this)
               (#?(:clj .read :cljs .-read) ^Flow other))
            (= (#?(:clj .write :cljs .-write) this)
               (#?(:clj .write :cljs .-write) ^Flow other))))))

  IFlow
  (close! [this]
    (close! this {}))
  (close! [this {:keys [drain] :or {drain true}}]
    (when (compare-and-set! closed false true)
      (a/close! (:channel write))
      (a/close! (:channel read))
      ;; Writers will not unblock on close
      (when drain
        (while (a/poll! (:channel write))))
      (doseq [handler (:close @handlers)]
        (handler this))
      (reset! handlers nil))
    this)
  (closed? [_this]
    (boolean @closed))
  (on-close [this f]
    (when @closed (f this))
    (swap! handlers update :close conj f))
  (on-error [_this f]
    (swap! handlers update :error conj f))
  (put! [this v]
    (let [puts (:puts pending)
          done (p/promise {:label (str "(put! " this ")")})]
      (swap! puts inc)
      (a/put! (:channel write) [v]
              (fn [result]
                (swap! puts dec)
                (if @closed
                  (p/reject! done false)
                  (if result
                    (p/resolve! done true)
                    (p/reject!  done false)))))
      done))
  (take! [this] (take! this nil))
  (take! [this default]
    (let [takes (:takes pending)
          done (p/promise {:label (str "(take! " this ")")})]
      (swap! takes inc)
      (a/take! (:channel read)
               (fn [v]
                 (swap! takes dec)
                 (p/resolve! done (if (nil? v) default (first v)))))
      done))

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Flow. label read write closed handlers pending meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-flow this)))]))

#?(:clj
   (defmethod print-method Flow
     [^Flow s w]
     (.write ^java.io.Writer w ^String (pr-flow s))))

#?(:clj
   (defmethod simple-dispatch Flow
     [^Flow s]
     (print (pr-flow s))))

(def ^:dynamic *error-handlers* [])

(defmacro with-error-handler
  [on-error & body]
  `(binding [*error-handlers* (conj *error-handlers* ~on-error)]
     ~@body))

(defonce ^:private default-error-handler (atom nil))

(defn set-default-on-error!
  [f]
  (reset! default-error-handler f))

(declare error-handlers error-handler)

(defn flow
  ([] (flow {}))
  ([{:keys [label buffer xform on-close _on-error] :as spec}]
   (let [error-handlers (error-handlers spec)
         flow (atom nil)
         chan (create-ch buffer xform (error-handler flow error-handlers))]
     (reset! flow (Flow. label
                         chan chan
                         (atom false)
                         (atom {:close (if on-close [on-close] [])
                                :error error-handlers})
                         {:puts (atom 0) :takes (atom 0)}
                         {})))))

(defn entangled
  ([] (entangled {} {}))
  ([flow-spec] (entangled flow-spec flow-spec))
  ([flow-spec-a flow-spec-b]
   (let [error-handlers-a (error-handlers flow-spec-a)
         error-handlers-b (error-handlers flow-spec-b)
         a (atom nil)
         b (atom nil)
         a-chan (create-ch (:buffer flow-spec-a) (:xform flow-spec-a)
                           (error-handler a error-handlers-a))
         b-chan (create-ch (:buffer flow-spec-b) (:xform flow-spec-b)
                           (error-handler b error-handlers-b))]
     (reset! a (Flow. (:label flow-spec-a)
                      a-chan b-chan
                      (atom false)
                      (atom {:close (if-let [on-close (:on-close flow-spec-a)] [on-close] [])
                             :error error-handlers-a})
                      {:puts (atom 0) :takes (atom 0)}
                      {}))
     (reset! b (Flow. (:label flow-spec-b)
                      b-chan a-chan
                      (atom false)
                      (atom {:close (if-let [on-close (:on-close flow-spec-b)] [on-close] [])
                             :error error-handlers-b})
                      {:puts (atom 0) :takes (atom 0)}
                      {}))
     (on-close @a (fn [_] (close! @b)))
     (on-close @b (fn [_] (close! @a)))
     [@a @b])))

(defn batched
  ([size latency]
   (batched {} size latency))
  ([{:keys [label buffer xform batch-buffer _error-handlers] :as spec} size latency]
   (let [error-handlers (error-handlers spec)
         flow (atom nil)
         value-ch (create-ch buffer xform (error-handler flow error-handlers))
         batch-ch (create-ch batch-buffer nil (error-handler flow error-handlers))
         limit (dec size)]
     (#?(:clj fiber :cljs a/go)
      (let [value-ch (:channel value-ch)
            batch-ch (:channel batch-ch)]
        (loop [batch [] t nil]
          (let [[v c] (#?(:clj a/alts!! :cljs a/alts!)
                       (if t [value-ch t] [value-ch]))]
            (cond
              (= c t)                  ; timeout
              (do
                (when (seq batch)
                  (#?(:clj a/>!! :cljs a/>!) batch-ch [batch]))
                (recur [] nil))

              (nil? v)                 ; value-ch has closed
              (do
                (when (seq batch)
                  (#?(:clj a/>!! :cljs a/>!) batch-ch [batch]))
                (a/close! batch-ch))

              (= (count batch) limit)
              (do
                (#?(:clj a/>!! :cljs a/>!) batch-ch [(conj batch (first v))])
                (recur [] nil))

              :else
              (recur (conj batch (first v))
                     (or t (a/timeout latency))))))))
     (reset! flow (Flow. label
                         batch-ch value-ch
                         (atom false)
                         (atom {:close [(fn [_] (a/close! (:channel value-ch)))]
                                :error error-handlers})
                         {:puts (atom 0) :takes (atom 0)}
                         {})))))

(defn flow?
  [flow]
  (satisfies? IFlow flow))

(def consume-stop ::consume-stop)

(defn consume
  [f ^Flow flow]
  (let [done (p/promise {:label (str "(consume " flow ")")})
        from-ch (:channel (#?(:clj .read :cljs .-read) flow))
        takes (:takes (#?(:clj .pending :cljs .-pending) flow))]
    (#?(:clj fiber :cljs a/go)
     {:label (str "(consume " flow ")")}
     (try
       (loop []
         (swap! takes inc)
         (let [x (#?(:clj a/<!! :cljs a/<!) from-ch)]
           (swap! takes dec)
           (when x
             (when-not (= consume-stop
                          (try
                            (f (first x))
                            (catch #?(:clj Throwable :cljs :default) e
                              (doseq [handler (-> flow :handlers :error)]
                                (try
                                  (handler flow e)
                                  (catch #?(:clj Throwable :cljs :default) _ nil))))))
               (recur)))))
       (p/resolve! done true)
       (catch #?(:clj Throwable :cljs :default) e
         (p/reject! done e))))
    done))

(defn put-all!
  [^Flow flow coll]
  (let [done (p/promise {:label (str "(put-all! " flow ")")})
        total (count coll)
        puts (:puts (#?(:clj .pending :cljs .-pending) flow))]
    (#?(:clj fiber :cljs a/go)
     {:label (str "(put-all! " flow ")")}
     (loop [coll (map vector coll)
            successful 0]
       (if (seq coll)
         (do
           (swap! puts inc)
           (let [written (#?(:clj a/>!! :cljs a/>!)
                          (:channel #?(:clj (.write flow)
                                       :cljs (.-write flow)))
                          (first coll))]
             (swap! puts dec)
             (if written
               (recur (rest coll) (inc successful))
               (p/resolve! done (= successful total)))))
         (p/resolve! done (= successful total)))))
    done))

(defn pipe
  ([^Flow src ^Flow dest] (pipe src dest nil))
  ([^Flow src ^Flow dest {:keys [xform
                                 close-src?
                                 close-dest?]
                          :or {close-src? false
                               close-dest? true}}]
   (when close-src?
     (on-close dest (fn [_] (close! src))))
   (when close-dest?
     (on-close src (fn [_] (close! dest))))
   (let [from (:channel (#?(:clj .read :cljs .-read) src))
         to (:channel (#?(:clj .write :cljs .-write) dest))
         xform (when xform (comp (map first) xform (map vector)))]
     #?(:clj
        (fiber {:label (str "(pipe " src " " dest ")")}
          (let [send (cond-> (fn
                               ([to] to)
                               ([to v]
                                (a/>!! to v)))
                       xform xform)]
            (loop []
              (when-let [v (a/<!! from)]
                (when (send to v)
                  (recur))))))
        :cljs
        (if xform
          (-> (a/pipe from (a/chan 1 xform) false)
              (a/pipe to false))
          (a/pipe from to false))))
   src))


;;; Private

(defn- error-handler
  [flow-atom error-handlers]
  (fn [e]
    (doseq [handler error-handlers]
      (try
        (handler @flow-atom e)
        (catch #?(:clj Throwable :cljs :default) _ nil)))))

(defn- error-handlers
  [{:keys [on-error] :as spec}]
  (if-let [error-handlers (cond
                            on-error
                            (conj *error-handlers* error-handlers)
                            @default-error-handler
                            (conj *error-handlers* @default-error-handler)
                            :else
                            (not-empty *error-handlers*))]
    error-handlers
    (throw (ex-info ":fluxus/flow missing error handler" {:spec spec}))))

(defn- create-ch
  [[type size :as buffer] xform error-handler]
  (let [a-buffer (when buffer
                   (if (and type (pos? size))
                     (case type
                       :fixed (a/buffer size)
                       :sliding (a/sliding-buffer size)
                       :dropping (a/dropping-buffer size))
                     (throw (ex-info ":fluxus/flow invalid buffer-spec" {:buffer buffer}))))
        a-xform (comp (map first) xform (map vector))
        a-chan (cond
                 (and buffer xform) (a/chan a-buffer a-xform error-handler)
                 xform (a/chan (a/buffer 1) a-xform error-handler)
                 buffer (a/chan a-buffer)
                 :else (a/chan))]
    {:buffer (when buffer [(first buffer) a-buffer])
     :channel a-chan}))

(defn- consumed
  [[type buffer]]
  #?(:clj
     (case type
       :fixed (.size ^LinkedList (.buf ^FixedBuffer buffer))
       :sliding (.size ^LinkedList (.buf ^SlidingBuffer buffer))
       :dropping (.size ^LinkedList (.buf ^DroppingBuffer buffer)))
     :cljs
     (case type
       :fixed (.-length ^RingBuffer (.-buf ^FixedBuffer buffer))
       :sliding (.-length ^RingBuffer (.-buf ^SlidingBuffer buffer))
       :dropping (.-length ^RingBuffer (.-buf ^DroppingBuffer buffer)))))

(defn- capacity
  [[type buffer]]
  #?(:clj
     (case type
       :fixed (.n ^FixedBuffer buffer)
       :sliding (.n ^SlidingBuffer buffer)
       :dropping (.n ^DroppingBuffer buffer))
     :cljs
     (case type
       :fixed (alength (.-arr ^RingBuffer (.-buf ^FixedBuffer buffer)))
       :sliding (alength (.-arr ^RingBuffer (.-buf ^SlidingBuffer buffer)))
       :dropping (alength (.-arr ^RingBuffer (.-buf ^DroppingBuffer buffer))))))

(defn- pr-flow
  [^Flow flow]
  (ust/format
   (str "#<fluxus/flow@" #?(:clj "0x%x" :cljs "%s") "%s %s>")
   (hash flow)
   (if-let [label (#?(:clj .label :cljs .-label) flow)]
     (str " " label)
     "")
   (if (closed? flow)
     "closed"
     (let [read (#?(:clj .read :cljs .-read) flow)
           write (#?(:clj .write :cljs .-write) flow)
           pending (#?(:clj .pending :cljs .-pending) flow)
           rb (:buffer read)
           wb (:buffer write)]
       (ust/format
        (str "[%s] :pending %s")
        (if (and rb wb)
          (let [pr-buffer (fn [buffer]
                            (ust/format "%s %d/%d" (first buffer) (consumed buffer) (capacity buffer)))]
            (if (not= read write)
              (ust/format "r[%s] w[%s]" (pr-buffer rb) (pr-buffer wb))
              (pr-buffer rb)))
          "")
        (pr-str (map-vals deref pending)))))))
