;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fluxus.promise
  (:refer-clojure :exclude [catch finally promise realized?])
  (:require [utilis.string :as ust]
            #?(:cljs [utilis.js :as j])
            #?(:clj [clojure.pprint :refer [simple-dispatch]]))
  #?(:clj (:import [clojure.lang IObj IMeta IDeref IBlockingDeref])))

#?(:cljs (set! *warn-on-infer* true))

(defprotocol IPromise
  (realized? [p])
  (resolved? [p])
  (rejected? [p])

  (resolve! [p v])
  (reject!  [p e])

  (then    [p f])
  (catch   [p f])
  (finally [p f]))

(declare register-handler run-handlers pr-promise throw-on-realized)

(deftype Promise [label backend handlers meta-map]
  Object
  (toString
    [^Promise this]
    (pr-promise this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash)
    [_]
    (hash [:fluxus/promise backend]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv)
    [^Promise this other]
    (boolean
     (when (instance? Promise other)
       (= (#?(:clj .backend :cljs .-backend) this)
          (#?(:clj .backend :cljs .-backend) ^Promise other)))))

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Promise. label backend handlers meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  IDeref
  (#?(:clj deref :cljs -deref)
    [^Promise #?(:clj _ :cljs this)]
    (let [[status value] #?(:clj (clojure.core/deref backend)
                            :cljs (if (realized? this)
                                    (clojure.core/deref backend)
                                    (throw (ex-info ":fluxus/promise unrealized" {:promise this}))))]
      (cond-> value (= :rejected status) throw)))
  #?@(:clj
      [IBlockingDeref
       (deref
        [_ timeout-ms timeout-val]
        (let [result (clojure.core/deref backend timeout-ms ::timeout)]
          (if (= result ::timeout)
            timeout-val
            (let [[status value] result]
              (cond-> value (= :rejected status) throw)))))])

  IPromise
  (realized?
    [_]
    #?(:clj (clojure.core/realized? backend)
       :cljs (boolean @backend)))
  (resolved?
    [^Promise this]
    (and (realized? this) (= :resolved (first (clojure.core/deref backend)))))
  (rejected?
    [^Promise this]
    (and (realized? this) (= :rejected (first (clojure.core/deref backend)))))
  (resolve!
    [^Promise this v]
    (locking backend
      (throw-on-realized this)
      (#?(:clj deliver :cljs reset!) backend [:resolved v])
      (run-handlers this))
    this)
  (reject!
    [^Promise this e]
    (locking backend
      (throw-on-realized this)
      (#?(:clj deliver :cljs reset!) backend [:rejected e])
      (run-handlers this))
    this)
  (then
    [^Promise this f]
    (register-handler this :then f))
  (catch
    [^Promise this f]
    (register-handler this :catch f))
  (finally
    [^Promise this f]
    (register-handler this :finally f))

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-promise this)))]))

#?(:clj
   (defmethod print-method Promise
     [^Promise p w]
     (.write ^java.io.Writer w ^String (pr-promise p))))

#?(:clj
   (defmethod simple-dispatch Promise
     [^Promise p]
     (print (pr-promise p))))

(declare promise-new)

(defn promise
  ([] (promise {}))
  ([{:keys [label]}]
   (let [p (Promise. label
                     #?(:clj  (clojure.core/promise)
                        :cljs (clojure.core/atom nil))
                     (atom
                      {:then []
                       :catch []
                       :finally []})
                     {})]
     #?(:clj p
        :cljs (j/assoc! p
                        :then (partial then p)
                        :catch (partial catch p)
                        :finally (partial finally p))))))

(defn promise?
  [p]
  (satisfies? IPromise p))

(defn resolved!
  ([v]
   (-> (promise) (resolve! v)))
  ([v {:keys [_label] :as opts}]
   (-> (promise opts) (resolve! v))))

(defn rejected!
  ([v]
   (-> (promise) (reject! v)))
  ([v {:keys [_label] :as opts}]
   (-> (promise opts) (reject! v))))

(defn all
  [promises]
  (let [p-result (promise)
        counter (atom 0)]
    (if (not-empty promises)
      (doseq [p promises]
        (let [handle (fn [_]
                       (when (= (count promises) (swap! counter inc))
                         (resolve! p-result promises)))]
          (-> p (then handle) (catch handle))))
      (resolve! p-result nil))
    p-result))

(defn any
  [promises]
  (let [p-result (promise)]
    (if (not-empty promises)
      (doseq [p promises]
        (let [handle (fn [_] (resolve! p-result p))]
          (-> p (then handle) (catch handle))))
      (resolve! p-result nil))
    p-result))


;;; Private

(defn- register-handler
  [^Promise this type f]
  (locking (#?(:clj .backend :cljs .-backend) this)
    (let [result-p (promise)]
      (swap! (#?(:clj .handlers :cljs .-handlers) this)
             update type conj [result-p f])
      (when (realized? this)
        (run-handlers this))
      result-p)))

(defn- run-handler
  [result-p f value]
  (try
    (let [callback-v (if (= ::no-value value) (f) (f value))]
      (if (promise? callback-v)
        ;; Ensure that a promise is not returned from
        ;; the handlers resulting in an infinite loop
        (-> callback-v
            (then (comp (constantly nil) (partial resolve! result-p)))
            (catch (comp (constantly nil) (partial reject! result-p))))
        (resolve! result-p callback-v)))
    (catch #?(:clj Exception :cljs :default) e
      (reject! result-p e))))

(defn- run-handlers
  [^Promise this]
  (let [handlers (#?(:clj .handlers :cljs .-handlers) this)
        [status value] (-> (#?(:clj .backend :cljs .-backend) this)
                           clojure.core/deref)]
    (try
      (doseq [[result-p handler] (-> handlers clojure.core/deref
                                     :then)]
        (if (= :resolved status)
          (run-handler result-p handler value)
          (reject! result-p value)))
      (doseq [[result-p handler] (-> handlers clojure.core/deref
                                     :catch)]
        (if (= :rejected status)
          (run-handler result-p handler value)
          (resolve! result-p value)))
      (finally
        (doseq [[result-p handler] (-> handlers clojure.core/deref
                                       :finally)]
          (run-handler result-p handler ::no-value))
        (reset! handlers nil)))))

(defn- throw-on-realized
  [promise]
  (when (realized? promise)
    (throw (ex-info ":fluxus/promise already realized"
                    {:promise promise}))))

(defn- pr-promise
  [^Promise p]
  (ust/format
   (str "#<fluxus/promise@" #?(:clj "0x%x" :cljs "%s") "%s%s>")
   (hash p)
   (if-let [label (#?(:clj .label
                      :cljs .-label) p)]
     (str "[" label "] ")
     " ")
   (if (realized? p)
     (pr-str (last
              (clojure.core/deref
               (#?(:clj .backend
                   :cljs .-backend) p))))
     "...")))
