;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fluxus.fiber
  (:require [fluxus.promise :as p])
  (:import [clojure.lang Var IObj IMeta IDeref]
           [java.util.concurrent Executors]
           [java.lang Thread InterruptedException]))

(defonce ^:private fiber-counter (atom 0))

(defprotocol IFiber
  (running? [f])
  (interrupted? [f])

  (interrupt! [f])

  (then [f callback])
  (catch [f callback])
  (finally [f callback]))

(declare pr-fiber)

(deftype Fiber [^Thread thread result exit-reason meta-map]
  Object
  (toString
    [^Fiber this]
    (pr-fiber this))
  (hashCode
    [_]
    (hash [:fluxus/fiber thread]))
  (equals
    [^Fiber this other]
    (boolean
     (when (instance? Fiber other)
       (= (.thread this)
          (.thread ^Fiber other)))))

  IObj
  (withMeta
    [_ meta-map]
    (Fiber. thread result exit-reason meta-map))

  IMeta
  (meta
    [_]
    meta-map)

  IDeref
  (deref
    [_]
    (clojure.core/deref result))

  IFiber
  (running?
    [_]
    (not (boolean exit-reason)))
  (interrupted?
    [_]
    (= :interrupted @exit-reason))

  (interrupt!
    [_]
    (.interrupt thread))

  (then
    [_ callback]
    (p/then result callback))
  (catch
      [_ callback]
      (p/catch result callback))
  (finally
    [_ callback]
    (p/finally result callback)))

(defmethod print-method Fiber [^Fiber p w]
  (.write ^java.io.Writer w ^String (pr-fiber p)))

(defmacro fiber
  [& body]
  `(let [captured-bindings# (Var/getThreadBindingFrame)
         result# (p/promise)
         exit# (atom nil)]
     (Fiber.
      (java.lang.Thread/startVirtualThread
       (^:once fn* []
        (Var/resetThreadBindingFrame captured-bindings#)
        (.setName (Thread/currentThread)
                  ~(str "fluxus.fiber/" (swap! fiber-counter inc)))
        (try
          (p/resolve! result# (do ~@body))
          (reset! exit# :normal)
          (catch InterruptedException e#
            (p/reject! result# e#)
            (reset! exit# :interrupted))
          (catch Throwable e#
            (p/reject! result# e#)
            (reset! exit# :error)))))
      result#
      exit#
      {})))

(defmacro concurrently
  [& body]
  (let [executor (gensym)]
    `(with-open [~executor (Executors/newThreadPerTaskExecutor
                            (-> (Thread/ofVirtual)
                                (.factory)))]
       ~@(map (fn [form]
                `(.submit ~executor
                          (vary-meta (^:once fn* [] ~form)
                                     assoc :tag `Callable))) body))))

;;; Private

(defn pr-fiber
  [^Fiber f]
  (format "#<fluxus/fiber@0x%x %s>"
          (hash f)
          (if (not (running? f))
            (pr-str (if (interrupted? f)
                      ":interrupted"
                      (clojure.core/deref f)))
            "...")))
