;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fluxus.flow
  (:require [fluxus.promise :as p]
            #?(:clj [fluxus.thread :refer [thread]])
            [utilis.map :refer [map-vals]]
            #?(:clj  [clojure.core.async :as a]
               :cljs [cljs.core.async :as a])
            [utilis.string :as ust])
  #?(:clj (:import [java.util LinkedList]
                   [clojure.core.async.impl.buffers
                    FixedBuffer SlidingBuffer DroppingBuffer]
                   [clojure.lang IObj IMeta])))

#?(:cljs (set! *warn-on-infer* true))

(defprotocol IFlow
  (close!  [flow])
  (closed? [flow])

  (on-close [flow f])
  (on-error [flow f])

  (put!  [flow v])
  (take! [flow] [s default]))

(declare pr-flow create-ch)

(deftype Flow [read write closed handlers pending meta-map]
  Object
  (toString
    [^Flow this]
    (pr-flow this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash)
    [_]
    (hash [:fluxus/flow read]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv)
    [^Flow this other]
    (boolean
     (when (instance? Flow other)
       (and (= (#?(:clj .read :cljs (.-read this)) this)
               (#?(:clj .read :cljs (.-read this)) ^Flow other))
            (= (#?(:clj .write :cljs (.-write this)) this)
               (#?(:clj .write :cljs (.-write this)) ^Flow other))))))

  IFlow
  (close! [this]
    (when (locking write
            (when (not @closed)
              (a/close! (:channel write))
              (reset! closed true)))
      (doseq [handler (:close @handlers)]
        (handler this))
      (reset! handlers nil))
    this)
  (closed? [_this]
    (boolean @closed))
  (on-close [_this f]
    (swap! handlers update :close conj f))
  (on-error [_this f]
    (swap! handlers update :error conj f))
  (put! [_this v]
    (let [puts (:puts pending)
          done (p/promise)]
      (swap! puts inc)
      (a/put! (:channel write) [v] (fn [result]
                                     (swap! puts dec)
                                     (p/resolve! done (boolean result))))
      done))
  (take! [this] (take! this nil))
  (take! [_this default]
    (let [takes (:takes pending)
          value (p/promise)]
      (swap! takes inc)
      (a/take! (:channel read)
               (fn [v]
                 (swap! takes dec)
                 (p/resolve! value (or (first v) default))))
      value))

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Flow. read write closed handlers pending meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-flow this)))]))

#?(:clj
   (defmethod print-method Flow [^Flow s w]
     (.write ^java.io.Writer w ^String (pr-flow s))))

(defonce ^:private default-error-handler (atom nil))

(defn set-default-on-error!
  [f]
  (reset! default-error-handler f))

(defn flow
  ([] (flow {}))
  ([{:keys [buffer xform on-error]
     :or {on-error @default-error-handler}}]
   (when-not on-error
     (throw (#?(:clj Exception. :cljs js/Error.)
             ":fluxus/flow missing error handler")))
   (let [chan (create-ch buffer xform)]
     (Flow. chan chan
            (atom false)
            (atom {:close []
                   :error [on-error]})
            {:puts (atom 0) :takes (atom 0)}
            {}))))

(defn entangled
  ([] (entangled {} {}))
  ([flow-spec-a flow-spec-b]
   (let [a-on-error (or (:on-error flow-spec-a) @default-error-handler)
         b-on-error (or (:on-error flow-spec-b) @default-error-handler)]
     (when-not a-on-error
       (throw (ex-info ":fluxus/flow missing error handler" {:spec flow-spec-a})))
     (when-not b-on-error
       (throw (ex-info ":fluxus/flow missing error handler" {:spec flow-spec-b})))
     (let [a-chan (create-ch (:buffer flow-spec-a) (:xform flow-spec-a))
           b-chan (create-ch (:buffer flow-spec-b) (:xform flow-spec-b))
           a (Flow. a-chan b-chan
                    (atom false)
                    (atom {:close []
                           :error [a-on-error]})
                    {:puts (atom 0) :takes (atom 0)}
                    {})
           b (Flow. b-chan a-chan
                    (atom false)
                    (atom {:close []
                           :error [b-on-error]})
                    {:puts (atom 0) :takes (atom 0)}
                    {})]
       (on-close a (fn [_] (close! b)))
       (on-close b (fn [_] (close! a)))
       [a b]))))

(defn batched
  ([size latency]
   (batched {} size latency))
  ([{:keys [buffer xform batch-buffer on-error]
     :or {on-error @default-error-handler}} size latency]
   (let [value-ch (create-ch buffer xform)
         batch-ch (create-ch batch-buffer nil)
         limit (dec size)]
     (#?(:clj thread :cljs a/go)
      (let [value-ch (:channel value-ch)
            batch-ch (:channel batch-ch)]
        (loop [batch [] t nil]
          (let [[v c] (#?(:clj a/alts!! :cljs a/alts!)
                       (if t [value-ch t] [value-ch]))]
            (cond
              (= c t) ; timeout
              (do
                (when (seq batch)
                  (#?(:clj a/>!! :cljs a/>!) batch-ch [batch]))
                (recur [] nil))

              (nil? v) ; value-ch has closed
              (do
                (when (seq batch)
                  (#?(:clj a/>!! :cljs a/>!) batch-ch [batch]))
                (a/close! batch-ch))

              (= (count batch) limit)
              (do
                (#?(:clj a/>!! :cljs a/>!) batch-ch [(conj batch (first v))])
                (recur [] nil))

              :else
              (recur (conj batch (first v))
                     (or t (a/timeout latency))))))))
     (Flow. batch-ch value-ch
            (atom false)
            (atom {:close [(fn [_] (a/close! (:channel value-ch)))]
                   :error [on-error]})
            {:puts (atom 0) :takes (atom 0)}
            {}))))

(defn flow?
  [flow]
  (satisfies? IFlow flow))

(defn consume
  [f ^Flow flow]
  (let [from-ch (:channel (#?(:clj .read :cljs .-read) flow))
        handlers (:error @(#?(:clj .handlers :cljs .-handlers) flow))]
    (#?(:clj thread :cljs a/go)
     (loop []
       (when-let [x (#?(:clj a/<!! :cljs a/<!) from-ch)]
         (try
           (f (first x))
           (catch #?(:clj Throwable :cljs :default) e
             (doseq [handler handlers]
               (try
                 (handler flow e)
                 (catch #?(:clj Exception :cljs :default) _)))))
         (recur)))))
  flow)

(defn put-all!
  [flow coll]
  (let [result (p/promise)]
    (#?(:clj thread :cljs a/go)
     (#?(:clj a/<!! :cljs a/<!)
      (#?(:clj a/onto-chan!! :cljs a/onto-chan!)
       (:channel #?(:clj (.write ^Flow flow)
                    :cljs (.-write flow)))
       (map vector coll) false))
     (p/resolve! result true))
    result))

(defn pipe
  ([^Flow src ^Flow dest] (pipe src dest nil))
  ([^Flow src ^Flow dest {:keys [xform
                                 close-src?
                                 close-dest?]
                          :or {close-src? false
                               close-dest? true}}]
   (when close-src?
     (on-close dest (fn [_] (close! src))))
   (when close-dest?
     (on-close src (fn [_] (close! dest))))
   (let [from (:channel (#?(:clj .read :cljs .-read) src))
         to (:channel (#?(:clj .write :cljs .-write) dest))
         recv (fn [from]
                (#?(:clj a/<!! :cljs a/<!) from))
         send (fn
                ([to] to)
                ([to unwrapped-v]
                 (#?(:clj a/>!! :cljs a/>!) to [unwrapped-v])))
         send (if xform (xform send) send)]
     (#?(:clj thread :cljs a/go)
      (loop []
        (let [v (recv from)]
          (when (and (not (nil? v))
                     (send to (first v))) ; unwrap for xform
            (recur))))))
   src))


;;; Private

(defn- create-ch
  [[type size :as buffer] xform]
  (let [a-buffer (when buffer
                   (if (and type (pos? size))
                     (case type
                       :fixed (a/buffer size)
                       :sliding (a/sliding-buffer size)
                       :dropping (a/dropping-buffer size))
                     (throw (ex-info ":fluxus/flow invalid buffer-spec" {:buffer buffer}))))
        a-chan (cond
                 (and buffer xform) (a/chan a-buffer (comp (map first) xform (map vector)))
                 xform (a/chan (a/buffer 1) (comp (map first) xform (map vector)))
                 buffer (a/chan a-buffer)
                 :else (a/chan))]
    {:buffer (when buffer [(first buffer) a-buffer])
     :channel a-chan}))

(defn- consumed
  [[type buffer]]
  #?(:clj (case type
            :fixed (.size ^LinkedList (.buf ^FixedBuffer buffer))
            :sliding (.size ^LinkedList (.buf ^SlidingBuffer buffer))
            :dropping (.size ^LinkedList (.buf ^DroppingBuffer buffer)))
     :cljs (.-length (.-buf buffer))))

(defn- capacity
  [[type buffer]]
  #?(:clj (case type
            :fixed (.n ^FixedBuffer buffer)
            :sliding (.n ^SlidingBuffer buffer)
            :dropping (.n ^DroppingBuffer buffer))
     :cljs (alength (.-arr (.-buf buffer)))))

(defn- pr-flow
  [^Flow flow]
  (ust/format
   (str "#<fluxus/flow@" #?(:clj "0x%x" :cljs "%s") " %s>")
   (hash flow)
   (if (closed? flow)
     "closed"
     (let [read (#?(:clj .read :cljs .-read) flow)
           write (#?(:clj .write :cljs .-write) flow)
           pending (#?(:clj .pending :cljs .-pending) flow)]
       (ust/format
        (str "[%s] :pending %s>")
        (let [pr-buffer (fn [buffer]
                          (ust/format "%s %d/%d" (first buffer) (consumed buffer) (capacity buffer)))]
          (let [rb (:buffer read)
                wb (:buffer write)]
            (if (not= read write)
              (str
               (when rb
                 (str "r[" (pr-buffer rb) "]"))
               (when (and rb wb) " ")
               (when wb
                 (str "w[" (pr-buffer wb) "]")))
              (str
               (when-let [rb (:buffer read)]
                 (pr-buffer rb))))))
        (pr-str (map-vals deref pending)))))))
