;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fluxus.thread
  (:require [fluxus.promise :as p])
  (:import [clojure.lang Var]
           [java.util.concurrent Executors]
           [java.lang Thread]))

(defonce thread-counter (atom 0))

(defmacro thread
  "Executes the body in a virtual thread, returning immediately to the
  calling thread. Returns a promise which will receive the result of
  the body when completed."
  [& body]
  `(let [result# (p/promise)
         captured-bindings# (Var/getThreadBindingFrame)]
     (Thread/startVirtualThread
      (^:once fn* []
       (Var/resetThreadBindingFrame captured-bindings#)
       (.setName (Thread/currentThread)
                 (str "fluxus-t-" (swap! thread-counter inc)))
       (try
         (p/resolve! result# (do ~@body))
         (catch Throwable e#
           (p/reject! result# e#)))))
     result#))

(defmacro concurrently
  [& body]
  (let [executor (gensym)]
    `(with-open [~executor (Executors/newThreadPerTaskExecutor
                            (-> (Thread/ofVirtual)
                                (.name "fluxus-c-" 0)
                                (.factory)))]
       ~@(map (fn [form]
                `(.submit ~executor
                          (vary-meta (^:once fn* [] ~form)
                                     assoc :tag `Callable))) body))))
