;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fluxus.actor
  (:require #?(:clj [fluxus.thread :refer [thread]])
            #?(:clj  [clojure.core.async :as a]
               :cljs [cljs.core.async :as a])
            [clojure.core.match :refer [match]]
            [utilis.coll :refer [nth-last]]
            [utilis.string :as ust])
  #?(:clj (:import [clojure.lang IObj IMeta]
                   [java.util UUID]
                   [java.lang IllegalArgumentException])))

(declare pr-actor)

(defprotocol IActor
  (! [a msg]))

(deftype Actor [fiber mailbox messages links meta-map]
  Object
  (toString
    [^Actor this]
    (pr-actor this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash)
    [_]
    (hash [:fluxus/actor mailbox]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv)
    [^Actor this other]
    (boolean
     (when (instance? Actor other)
       (= (#?(:clj .mailbox :cljs (.-mailbox this)) this)
          (#?(:clj .mailbox :cljs (.-mailbox this)) ^Actor other)))))

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Actor. fiber mailbox messages links meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  IActor
  (! [this msg]
    (#?(:clj a/>!! :cljs a/>!) mailbox [msg])
    msg)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-actor this)))]))

#?(:clj
   (defmethod print-method Actor [^Actor p w]
     (.write ^java.io.Writer w ^String (pr-actor p))))

(def ^:dynamic ^Actor *self*)

(defn self
  []
  *self*)

(defmacro spawn
  [& body]
  (let [opt-keys #{:name :link :trap :mailbox-size :overflow-policy}
        opts (->> body
                  (partition-all 2)
                  (map vec)
                  (take-while (fn [[k v :as pair]]
                                (and (= 2 (count pair))
                                     (get opt-keys k))))
                  (into {}))
        body (drop (* 2 (count opts)) body)
        mailbox-size (get opts :mailbox-size 1024)]
    `(let [mailbox# (a/chan ~mailbox-size)
           messages# (atom [])
           self# (Actor. (atom nil) mailbox# messages#
                         (atom ~(if-let [link (get opts :link)]
                                  #{link}
                                  #{})) nil)
           fiber# (#?(:clj thread :cljs a/go)
                   (binding [*self* self#]
                     ~@body))]
       (reset! (#?(:clj .fiber :cljs .-fiber) self#) fiber#)
       self#)))

(defmacro spawn-link
  [& body]
  `(if-not (boolean (instance? Actor (self)))
     (throw (ex-info "must be called from actor" {}))
     (spawn :link (self) ~@body)))

;; TODO: Attempting to link to a non-existent process kills the linker
(defn link!
  ([other]
   (link! (self) other))
  ([actor0 actor1]
   (swap! (#?(:clj .links :cljs .-links) actor0) conj actor1)
   (swap! (#?(:clj .links :cljs .-links) actor1) conj actor0)
   nil))

(defn unlink!
  ([other]
   (unlink! (self) other))
  ([actor0 actor1]
   (swap! (#?(:clj .links :cljs .-links) actor0) disj actor1)
   (swap! (#?(:clj .links :cljs .-links) actor1) disj actor0)
   nil))

(defn maketag
  []
  (str #?(:clj (UUID/randomUUID) :cljs (random-uuid))))

(defmacro receive
  [& body]
  (let [[body after] (if (and (>= (count body) 3)
                              (= :timeout (nth-last body 2)))
                       (split-at (- (count body) 3) body)
                       [body nil])
        mailbox (gensym "mailbox_")
        result (gensym "result_")]
    `(loop []
       (let [messages# (#?(:clj .messages :cljs .-messages) *self*)
             ~result (atom ::no-match)
             match# (fn [msg#]
                      (try
                        (reset! ~result (match (first msg#) ~@body))
                        true
                        (catch IllegalArgumentException e#
                          (if (= (.getMessage e#)
                                 (str "No matching clause: " (first msg#)))
                            false
                            (throw e#)))))]
         (swap!
          messages#
          (fn [messages#]
            (->> (range 0 (count messages#))
                 (map (fn [i#] (drop i# messages#)))
                 (reduce (fn [seen# [msg# & rest#]]
                           (if (match# msg#)
                             (reduced (into seen# rest#))
                             (conj seen# msg#))) []))))
         (when (= ::no-match @~result)
           (let [~mailbox (#?(:clj .mailbox :cljs .-mailbox) *self*)
                 new-msg# ~(if after
                             `(let [timeout-ch# (a/timeout ~(second after))
                                    [v# ch#] (#?(:clj a/alts!! :cljs a/alts!)
                                              [~mailbox timeout-ch#])]
                                (if (= timeout-ch# ch#)
                                  (do (reset! ~result ~(last after)) nil)
                                  v#))
                             `(#?(:clj a/<!! :cljs a/<!) ~mailbox))]
             (when (and new-msg# (not (match# new-msg#)))
               (swap! messages# conj new-msg#))))
         (if (= ::no-match @~result)
           (recur)
           @~result)))))

#_(let [res (atom [])
        actor (spawn
               (loop []
                 (receive
                  [:foo x] (do
                             (swap! res conj x)
                             (receive
                              [:baz z] (swap! res conj z)))
                  [:bar y] (swap! res conj y))
                 (recur)))]
    (! actor [:foo 1])
    (! actor [:bar 2])
    (! actor [:baz 3])
    (Thread/sleep 100)
    @res) ; => [1 3 2]

#_(let [tag (maketag)
        adder (spawn
               (loop []
                 (receive
                  [from tag [:add a b]] (! from [tag [:sum (+ a b)]]))
                 (recur)))]
    (spawn
     (! adder [(self) tag [:add 5 6]])
     (println "sum:" (receive
                      [tag [:sum sum]] sum
                      :timeout 100 :timeout))))

#_(let [adder (spawn
               (loop []
                 (receive
                  [from tag [:add a b]] (! from [tag [:sum (+ a b)]]))
                 (recur)))
        computer (spawn
                  (loop []
                    (receive
                     [from tag [:compute a b c d]] (let [tag1 (maketag)]
                                                     (! adder [(self) tag1 [:add (* a b) (* c d)]])
                                                     (receive
                                                      [tag1 [:sum sum]]  (! from [tag [:result sum]])
                                                      :timeout 10          (! from [tag [:error "timeout!"]])
                                                      ))
                     msg (println "Unknown message: " msg))
                    (recur)))
        nums (take 20 (repeatedly #(rand-int 10)))
        curious (spawn
                 (loop [nums nums]
                   (when (seq nums)
                     (let [[a b c d] (take 4 nums)
                           tag       (maketag)]
                       (! computer [(self) tag [:compute a b c d]])
                       (receive
                        [tag [:result res]]  (println a b c d "->" res)
                        [tag [:error error]] (println "ERROR: " a b c d "->" error)
                        msg (println "Unexpected message H" msg))
                       (recur (drop 4 nums))))))]
    (Thread/sleep 5000))

;;; Private

(defn- pr-actor
  [^Actor a]
  (ust/format (str "#<fluxus/actor@" #?(:clj "0x%x" :cljs "%s") " [%d]>")
              (hash a)
              (count @(#?(:clj .messages :cljs .-messages) a))))
