;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fluxus.promise
  (:refer-clojure :exclude [catch finally promise realized?])
  (:require [utilis.string :as ust]
            #?(:cljs [utilis.js :as j]))
  #?(:clj (:import [clojure.lang IObj IMeta IDeref])))

(defprotocol IPromise
  (realized? [p])
  (resolved? [p])
  (rejected? [p])

  (resolve! [p v])
  (reject!  [p e])

  (then    [p f])
  (catch   [p f])
  (finally [p f]))

(declare promise pr-promise throw-on-realized)

(deftype Promise [backend handlers meta-map]
  Object
  (toString
    [^Promise this]
    (pr-promise this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash)
    [_]
    (hash [:fluxus/promise backend]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv)
    [^Promise this other]
    (boolean
     (when (instance? Promise other)
       (= (#?(:clj .backend :cljs (.-backend this)) this)
          (#?(:clj .backend :cljs (.-backend this)) ^Promise other)))))

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Promise. backend handlers meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  IDeref
  (#?(:clj deref :cljs -deref)
    [^Promise this]
    (let [[status value] #?(:clj (clojure.core/deref backend)
                            :cljs (if (realized? this)
                                    (clojure.core/deref backend)
                                    (throw (js/Error. "unrealized"))))]
      (cond-> value (= :rejected status) throw)))

  IPromise
  (realized?
    [_]
    #?(:clj (clojure.core/realized? backend)
       :cljs (boolean @backend)))
  (resolved?
    [^Promise this]
    (and (realized? this) (= :resolved (first (clojure.core/deref backend)))))
  (rejected?
    [^Promise this]
    (and (realized? this) (= :rejected (first (clojure.core/deref backend)))))
  (resolve!
    [^Promise this v]
    (locking backend
      (throw-on-realized this)
      (#?(:clj deliver :cljs reset!) backend [:resolved v])
      (doseq [[result-p handler] (:resolved @handlers)]
        (try
          (resolve! result-p (handler v))
          (catch #?(:clj Exception :cljs :default) e
            (reject! result-p e))))
      (reset! handlers nil))
    this)
  (reject!
    [^Promise this e]
    (locking backend
      (throw-on-realized this)
      (#?(:clj deliver :cljs reset!) backend [:rejected e])
      (doseq [handler (:rejected @handlers)]
        (try
          (handler e)
          (catch #?(:clj Exception :cljs :default) _)))
      (doseq [[result-p _] (:resolved @handlers)]
        (try
          (reject! result-p e)
          (catch #?(:clj Exception :cljs :default) _)))
      (reset! handlers nil))
    this)
  (then
    [^Promise this f]
    (locking backend
      (let [result (promise)]
        (if (realized? this)
          (let [[status value] (clojure.core/deref backend)]
            (when (= :resolved status)
              (resolve! result (f value))))
          (swap! handlers update :resolved conj [result f]))
        result)))
  (catch
      [^Promise this f]
      (locking backend
        (if (realized? this)
          (let [[status value] (clojure.core/deref backend)]
            (when (= :rejected status)
              (f value)))
          (swap! handlers update :rejected conj f))
        this))
  (finally
    [^Promise this f]
    (locking backend
      (if (realized? this)
        (let [[_status value] (clojure.core/deref backend)]
          (f value))
        (swap! handlers (fn [handlers]
                          (-> handlers
                              (update :resolved conj [(promise) f])
                              (update :rejected conj f)))))
      this))

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-promise this)))]))

#?(:clj
   (defmethod print-method Promise [^Promise p w]
     (.write ^java.io.Writer w ^String (pr-promise p))))

(defn promise
  []
  (let [p (Promise. #?(:clj  (clojure.core/promise)
                       :cljs (clojure.core/atom nil))
                    (atom
                     {:resolved []
                      :rejected []})
                    {})]
    #?(:clj p
       :cljs (j/assoc! p
                       :then (partial then p)
                       :catch (partial catch p)
                       :finally (partial finally p)))))

(defn promise?
  [p]
  (satisfies? IPromise p))


;;; Private

(defn- throw-on-realized
  [promise]
  (when (realized? promise)
    (throw (ex-info "already realized" {:promise promise}))))

(defn- pr-promise
  [^Promise p]
  (ust/format (str "#<fluxus/promise@" #?(:clj "0x%x" :cljs "%s") " %s>")
              (hash p)
              (if (realized? p)
                (pr-str (last
                         (clojure.core/deref
                          (#?(:clj .backend
                              :cljs .-backend) p))))
                "...")))
