;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fluxus.flow
  (:require [fluxus.promise :as p]
            [utilis.map :refer [map-vals]]
            [utilis.timer :as timer]
            #?(:clj  [clojure.core.async :as a]
               :cljs [cljs.core.async :as a])
            [utilis.string :as ust])
  #?(:clj (:import [java.util LinkedList]
                   [clojure.core.async.impl.buffers
                    FixedBuffer SlidingBuffer DroppingBuffer]
                   [clojure.lang IObj IMeta])))

(defprotocol IFlow
  (close!  [flow])
  (closed? [flow])

  (on-close [flow f])
  (on-error [flow f])

  (put!  [flow v])
  (take! [flow] [s default])

  (entangled? [flow]))

(declare pr-flow create-ch)

(deftype Flow [read write closed handlers pending meta-map]
  Object
  (toString
    [^Flow this]
    (pr-flow this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash)
    [_]
    (hash [:fluxus/flow read]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv)
    [^Flow this other]
    (boolean
     (when (instance? Flow other)
       (and (= (#?(:clj .read :cljs (.-read this)) this)
               (#?(:clj .read :cljs (.-read this)) ^Flow other))
            (= (#?(:clj .write :cljs (.-write this)) this)
               (#?(:clj .write :cljs (.-write this)) ^Flow other))))))

  IFlow
  (close! [this]
    (locking write
      (when (not @closed)
        (a/close! (:channel write))
        (reset! closed true)
        (doseq [handler (:close @handlers)]
          (handler this))
        (reset! handlers nil)))
    this)
  (closed? [_this]
    (boolean @closed))
  (on-close [_this f]
    (swap! handlers update :close conj f))
  (on-error [_this f]
    (swap! handlers update :error conj f))
  (put! [_this v]
    (let [puts (:puts pending)
          done (p/promise)]
      (swap! puts inc)
      (a/put! (:channel write) [v] (fn [result]
                                     (swap! puts dec)
                                     (p/resolve! done (boolean result))))
      done))
  (take! [this] (take! this nil))
  (take! [_this default]
    (let [takes (:takes pending)
          value (p/promise)]
      (swap! takes inc)
      (a/take! (:channel read)
               (fn [v]
                 (swap! takes dec)
                 (p/resolve! value (or (first v) default))))
      value))
  (entangled? [_] (not= read write))

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Flow. read write closed handlers pending meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-flow this)))]))

#?(:clj
   (defmethod print-method Flow [^Flow s w]
     (.write ^java.io.Writer w ^String (pr-flow s))))

(defonce ^:private default-error-handler (atom nil))

(defn set-default-on-error!
  [f]
  (reset! default-error-handler f))

(defn flow
  ([] (flow {}))
  ([{:keys [buffer xform on-error]
     :or {on-error @default-error-handler}}]
   (when-not on-error
     (throw (#?(:clj Exception. :cljs js/Error.)
             "Missing error handler")))
   (let [chan (create-ch buffer xform)]
     (Flow. chan chan
            (atom false)
            (atom {:close []
                   :error [on-error]})
            {:puts (atom 0) :takes (atom 0)}
            {}))))

(defn entangled
  ([] (entangled {} {}))
  ([flow-spec-a flow-spec-b]
   (let [a-on-error (or (:on-error flow-spec-a) @default-error-handler)
         b-on-error (or (:on-error flow-spec-b) @default-error-handler)]
     (when-not a-on-error
       (throw (ex-info "Missing error handler" {:spec flow-spec-a})))
     (when-not b-on-error
       (throw (ex-info "Missing error handler" {:spec flow-spec-b})))
     (let [a-chan (create-ch (:buffer flow-spec-a) (:xform flow-spec-a))
           b-chan (create-ch (:buffer flow-spec-b) (:xform flow-spec-b))
           a (Flow. a-chan b-chan
                    (atom false)
                    (atom {:close []
                           :error [a-on-error]})
                    {:puts (atom 0) :takes (atom 0)}
                    {})
           b (Flow. b-chan a-chan
                    (atom false)
                    (atom {:close []
                           :error [b-on-error]})
                    {:puts (atom 0) :takes (atom 0)}
                    {})]
       (on-close a (fn [_] (close! b)))
       (on-close b (fn [_] (close! a)))
       [a b]))))

(defn flow?
  [flow]
  (satisfies? IFlow flow))

(defn consume
  [f ^Flow flow]
  (let [from-ch (:channel (#?(:clj .read :cljs .-read) flow))
        handlers (:error @(#?(:clj .handlers :cljs .-handlers) flow))]
    (a/go-loop []
      (when-let [x (a/<! from-ch)]
        (try
          (f (first x))
          (catch #?(:clj Throwable :cljs :default) e
            (doseq [handler handlers]
              (try
                (handler flow e)
                (catch #?(:clj Exception :cljs :default) _)))))
        (recur))))
  flow)

(defn put-all!
  [flow coll]
  (let [result (p/promise)]
    (a/go
      (a/<! (a/onto-chan! (:channel #?(:clj (.write ^Flow flow)
                                       :cljs (.-write flow)))
                          (map vector coll) false))
      (p/resolve! result true))
    result))

(defn connect
  [^Flow src ^Flow dest]
  (on-close src (fn [_] (close! dest)))
  (on-close dest (fn [_] (close! src)))
  (consume (partial put! dest) src))

(defn batching-xform
  [size latency]
  (fn [rf]
    (let [batch (volatile! [])
          timer (volatile! nil)
          reset-batch (fn []
                        (locking batch
                          (let [items @batch]
                            (vreset! batch [])
                            items)))]
      (fn
        ([r]
         (locking batch
           (when @timer (timer/cancel @timer))
           (let [result (if (empty? @batch)
                          r
                          (unreduced (rf r (reset-batch))))]
             (rf result))))
        ([r x]
         (locking batch
           (vswap! batch conj x))
         (if (= size (count @batch))
           (do
             (vreset! timer (timer/run-after #(rf r (reset-batch)) latency))
             (rf r (reset-batch)))
           r))))))


;;; Private

(defn- create-ch
  [[type size :as buffer] xform]
  (let [a-buffer (when buffer
                   (if (and type (pos? size))
                     (case type
                       :fixed (a/buffer size)
                       :sliding (a/sliding-buffer size)
                       :dropping (a/dropping-buffer size))
                     (throw (ex-info "Invalid buffer-spec" {:buffer buffer}))))
        a-chan (cond
                 (and buffer xform) (a/chan a-buffer (comp (map first) xform (map vector)))
                 xform (a/chan (a/buffer 1) (comp (map first) xform (map vector)))
                 buffer (a/chan a-buffer)
                 :else (a/chan))]
    {:buffer (when buffer [(first buffer) a-buffer])
     :channel a-chan}))

(defn- consumed
  [[type buffer]]
  #?(:clj (case type
            :fixed (.size ^LinkedList (.buf ^FixedBuffer buffer))
            :sliding (.size ^LinkedList (.buf ^SlidingBuffer buffer))
            :dropping (.size ^LinkedList (.buf ^DroppingBuffer buffer)))
     :cljs (.-length (.-buf buffer))))

(defn- capacity
  [[type buffer]]
  #?(:clj (case type
            :fixed (.n ^FixedBuffer buffer)
            :sliding (.n ^SlidingBuffer buffer)
            :dropping (.n ^DroppingBuffer buffer))
     :cljs (alength (.-arr (.-buf buffer)))))

(defn- pr-flow
  [^Flow flow]
  (ust/format
   (str "#<fluxus/flow@" #?(:clj "0x%x" :cljs "%s") " %s>")
   (hash flow)
   (if (closed? flow)
     "closed"
     (let [read (#?(:clj .read :cljs .-read) flow)
           write (#?(:clj .write :cljs .-write) flow)
           pending (#?(:clj .pending :cljs .-pending) flow)]
       (ust/format
        (str "%s[%s] :pending %s>")
        (str (when (entangled? flow) "entangled "))
        (let [pr-buffer (fn [buffer]
                          (ust/format "%s %d/%d" (first buffer) (consumed buffer) (capacity buffer)))]
          (if (entangled? flow)
            (let [rb (:buffer read)
                  wb (:buffer write)]
              (str
               (when rb
                 (str "< " (pr-buffer rb)))
               (when (and rb wb) " ")
               (when wb
                 (str "> " (pr-buffer wb)))))
            (when-let [rb (:buffer read)]
              (pr-buffer rb))))
        (pr-str (map-vals deref pending)))))))
