;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fluxus.stream
  (:require [fluxus.deferred :as d]
            [utilis.map :refer [map-vals]]
            [utilis.string :as ust]
            #?(:cljs [utilis.js :as j])
            #?(:clj  [clojure.core.async :as a]
               :cljs [cljs.core.async :as a]))
  #?(:clj (:import [java.util LinkedList]
                   [clojure.core.async.impl.buffers
                    FixedBuffer SlidingBuffer DroppingBuffer])))

(declare pr-buffer pr-stream pr-paired-stream wrap unwrap)

(defprotocol IBuffer
  (consumed [buffer])
  (capacity [buffer]))

(deftype Buffer [type buffer]
  Object
  (toString [^Buffer this]
    (pr-buffer this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash) [_]
    (hash [:fluxus/buffer buffer]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv) [this other]
    (boolean
     (when (instance? Buffer other)
       (= (#?(:clj .buffer :cljs (j/get this :buffer)) this)
          (#?(:clj .buffer :cljs (j/get this :buffer)) ^Buffer other)))))

  IBuffer
  (consumed [_this]
    #?(:clj (case type
              :fixed (.size ^LinkedList (.buf ^FixedBuffer buffer))
              :sliding (.size ^LinkedList (.buf ^SlidingBuffer buffer))
              :dropping (.size ^LinkedList (.buf ^DroppingBuffer buffer)))
       :cljs (j/get-in buffer [:buf :length])))
  (capacity [_this]
    #?(:clj (case type
              :fixed (.n ^FixedBuffer buffer)
              :sliding (.n ^SlidingBuffer buffer)
              :dropping (.n ^DroppingBuffer buffer))
       :cljs (alength (j/get-in buffer [:buf :arr]))))

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-buffer this)))]))

#?(:clj
   (defmethod print-method Buffer [^Buffer s w]
     (.write ^java.io.Writer w ^String (pr-buffer s))))

(defn buffer
  [size]
  (Buffer. :fixed (a/buffer size)))

(defn sliding-buffer
  [size]
  (Buffer. :sliding (a/sliding-buffer size)))

(defn dropping-buffer
  [size]
  (Buffer. :dropping (a/dropping-buffer size)))

(defprotocol IStream
  (close! [s])
  (closed? [s])

  (on-close [s f])
  (on-error [s f])

  (put! [s v])
  (take! [s] [s default]))

(deftype Stream [^Buffer buffer chan handlers closed pending]
  Object
  (toString [^Stream this]
    (pr-stream this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash) [_]
    (hash [:fluxus/stream chan]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv) [this other]
    (boolean
     (when (instance? Stream other)
       (= (#?(:clj .chan :cljs (j/get this :chan)) this)
          (#?(:clj .chan :cljs (j/get this :chan)) ^Stream other)))))

  IStream
  (close! [this]
    (when (not @closed)
      (a/close! chan)
      (reset! closed true)
      (doseq [handler @(:close handlers)]
        (handler this))))
  (closed? [_this]
    (boolean @closed))
  (on-close [_this f]
    (swap! (:close handlers) conj f))
  (on-error [_this f]
    (swap! (:error handlers) conj f))
  (put! [_this v]
    (let [puts (:puts pending)
          done (d/deferred)]
      (swap! puts inc)
      (a/put! chan (wrap v)
              (fn [result]
                (swap! puts dec)
                (d/success! done (boolean result))))
      done))
  (take! [this] (take! this nil))
  (take! [_this default]
    (let [takes (:takes pending)
          value (d/deferred)]
      (swap! takes inc)
      (a/take! chan
               (fn [v]
                 (swap! takes dec)
                 (d/success! value (or (unwrap v) default))))
      value))

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-stream this)))]))

#?(:clj
   (defmethod print-method Stream [^Stream s w]
     (.write ^java.io.Writer w ^String (pr-stream s))))

(deftype PairedStream [^Stream output ^Stream input handlers closed]
  Object
  (toString [^PairedStream this]
    (pr-paired-stream this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash) [_]
    (hash [:fluxus/paired-stream output input]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv) [this other]
    (boolean
     (when (instance? PairedStream other)
       (and (= (#?(:clj .output :cljs (j/get this :output)) this)
               (#?(:clj .output :cljs (j/get this :output)) ^PairedStream other))
            (= (#?(:clj .input :cljs (j/get this :input)) this)
               (#?(:clj .input :cljs (j/get this :input)) ^PairedStream other))))))

  IStream
  (close! [this]
    (when (not @closed)
      (reset! closed true)
      (close! output)
      (close! input)
      (doseq [handler @(:close handlers)]
        (handler this))))
  (closed? [_this]
    (boolean @closed))
  (on-close [_this f]
    (swap! (:close handlers) conj f))
  (on-error [_this f]
    (swap! (:error handlers) conj f))
  (put! [_this v]
    (put! output v))
  (take! [this]
    (take! this nil))
  (take! [_this default]
    (take! input default))

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-paired-stream this)))]))

#?(:clj
   (defmethod print-method PairedStream [^PairedStream s w]
     (.write ^java.io.Writer w ^String (pr-paired-stream s))))

(defonce ^:private default-error-handler (atom nil))

(defn set-default-on-error!
  [f]
  (reset! default-error-handler f))

(defn stream
  [& {:keys [buffer on-error] :or {on-error @default-error-handler}}]
  (when-not on-error
    (throw (ex-info "[fluxus/stream] Missing error handler" {})))
  (Stream. buffer
           (if buffer
             (a/chan #?(:clj (.buffer ^Buffer buffer)
                        :cljs (j/get buffer :buffer)))
             (a/chan))
           {:close (atom [])
            :error (atom [on-error])}
           (atom false)
           {:puts (atom 0)
            :takes (atom 0)}))

(defn stream?
  [s]
  (satisfies? IStream s))

(declare get-ch)

(defn consume
  [f s]
  (let [from-ch (get-ch s :input)
        handlers (:error
                  #?(:clj (if (instance? PairedStream s)
                            (.handlers ^PairedStream s)
                            (.handlers ^Stream s))
                     :cljs (j/get s :handlers)))]
    (a/go-loop []
      (when-let [x (a/<! from-ch)]
        (try
          (f (unwrap x))
          (catch #?(:clj Throwable :cljs :default) e
            (doseq [handler @handlers]
              (handler s e))))
        (recur))))
  s)

(defn connect
  [from to & {:keys [close-to?] :or {close-to? true}}]
  (a/pipe (get-ch from :input)
          (get-ch to :output)
          close-to?)
  nil)

(defn batch-connect
  [from to size latency
   & {:keys [close-to?] :or {close-to? true}}]
  (let [from-ch (get-ch from :input)
        forward (fn [batch] (when (seq batch) (put! to batch)))]
    (a/go
      (loop [batch []]
        (if (< (count batch) size)
          (let [[v ch] (a/alts! [from-ch (a/timeout latency)])]
            (if (= from-ch ch)
              (if v
                (recur (conj batch (unwrap v)))
                (forward batch))
              (do (forward batch)
                  (recur []))))
          (do
            (forward batch)
            (recur []))))
      (when close-to? (close! to))))
  nil)

(defn paired-stream
  [output input]
  (let [paired (PairedStream. output input
                              {:close (atom [])
                               :error (atom [on-error])}
                              (atom false))]
    (on-close paired (fn [_] (close! output)))
    (on-close input (fn [_] (close! paired)))
    paired))


;;; Private

(defn- get-ch
  [s direction]
  (let [s (if (instance? PairedStream s)
            (case direction
              :output #?(:clj (.output ^PairedStream s)
                      :cljs (j/get s :output))
              :input  #?(:clj (.input ^PairedStream s)
                         :cljs (j/get s :input)))
            s)]
    (if (instance? PairedStream s)
      (get-ch s direction)
      #?(:clj (.chan ^Stream s) :cljs (j/get s :chan)))))

(defn- wrap
  [x]
  [x])

(defn- unwrap
  [x]
  (first x))

(defn- pr-buffer
  [^Buffer b]
  (let [type #?(:clj (.type b) :cljs (j/get b :type))]
    (ust/format (str "#<fluxus/Buffer@" #?(:clj "0x%x" :cljs "%s") " %s %d/%d>")
                (hash b)
                type
                (consumed b)
                (capacity b))))

(defn- pr-stream
  [^Stream s]
  (let [buffer #?(:clj (.buffer s) :cljs (j/get s :buffer))
        closed #?(:clj @(.closed s) :cljs @(j/get s :closed))
        pending #?(:clj (.pending s) :cljs (j/get s :pending))]
    (ust/format (str "#<fluxus/Stream@" #?(:clj "0x%x" :cljs "%s") " [%s] :closed? %s :pending %s>")
                (hash s)
                (if buffer (pr-buffer buffer) "")
                (str closed)
                (pr-str (map-vals deref pending)))))

(defn- pr-paired-stream
  [^PairedStream s]
  (ust/format (str "#<fluxus/PairedStream@" #?(:clj "0x%x" :cljs "%s") " :output %s :input %s>")
              (hash s)
              #?(:clj (.output s) :cljs (j/get s :output))
              #?(:clj (.input s) :cljs (j/get s :input))))
