;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fluxus.deferred
  (:refer-clojure :exclude [realized?])
  (:require [utilis.string :as ust]
            #?(:cljs [utilis.js :as j]))
  #?(:clj (:import [clojure.lang IDeref])))

(declare throw-on-realized pr-deferred)

(defprotocol IDeferred
  (success! [d v])
  (error! [d e])
  (realized? [d])
  (on-success [d f])
  (on-error [d f]))

(deftype Deferred [backend handlers]
  Object
  (toString [^Deferred this]
    (pr-deferred this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash) [_]
    (hash [:fluxus/deferred backend]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv) [this other]
    (= (#?(:clj (.backend this)
           :cljs (j/get this :backend)))
       (#?(:clj (.backend ^Deferred other)
           :cljs (j/get this :backend)))))

  IDeferred
  (success! [d v]
    (locking backend
      (throw-on-realized d)
      (#?(:clj deliver :cljs reset!) backend {:status :success :value v})
      (doseq [handler @(:success handlers)]
        (handler v)))
    d)
  (error! [d e]
    (locking backend
      (throw-on-realized d)
      (#?(:clj deliver :cljs reset!) backend {:status :error :value e})
      (doseq [handler @(:error handlers)]
        (handler e)))
    d)
  (realized? [d]
    #?(:clj (clojure.core/realized? backend)
       :cljs (boolean @backend)))
  (on-success [d f]
    (locking backend
      (swap! (:success handlers) conj f)
      (when (and (realized? d)
                 (= :success (:status (clojure.core/deref backend))))
        (f (:value (clojure.core/deref backend)))))
    d)
  (on-error [d f]
    (locking backend
      (swap! (:error handlers) conj f)
      (when (and (realized? d)
                 (= :error (:status (clojure.core/deref backend))))
        (f (:value (clojure.core/deref backend)))))
    d)

  IDeref
  (#?(:clj deref :cljs -deref)
    [_d]
    #?(:clj (:value (clojure.core/deref backend))
       :cljs (or
              (:value (clojure.core/deref backend))
              (throw (ex-info "[fluxus/deferred] Unrealized" {})))))

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-deferred this)))]))

#?(:clj
   (defmethod print-method Deferred [^Deferred s w]
     (.write ^java.io.Writer w ^String (pr-deferred s))))

(defn deferred
  []
  (let [d (Deferred. #?(:clj (promise) :cljs (atom nil))
            {:success (atom [])
             :error (atom [])})]
    #?(:clj d
       :cljs (js/Object.assign
              d
              #js {:then (fn [callback]
                           (try
                             (on-success d callback)
                             (catch :default e
                               (error! d e))))
                   :catch (fn [callback]
                            (on-error d callback))
                   :finally (fn [callback]
                              (on-success d callback)
                              (on-error d callback))}))))

(defn deferred?
  [d]
  (satisfies? IDeferred d))


;;; Private

(defn- throw-on-realized
  [deferred]
  (when (realized? deferred)
    (throw (ex-info "[fluxus/Deferred already delivered"
                    {:deferred deferred}))))

(defn- pr-deferred
  [^Deferred d]
  (let [backend #?(:clj (.backend d) :cljs (j/get d :backend))
        {:keys [status value]} (if (realized? d)
                                 (clojure.core/deref backend)
                                 {:status :pending})]
    (ust/format (str "#<fluxus/Deferred@" #?(:clj "0x%x" :cljs "%s") " :status %s :value %s>")
                (hash d)
                (str status)
                (pr-str value))))
