(ns orcl.testkit.core
  (:require [orcl.testkit.tests :as tests]
            [orcl.compiler :as compiler]
            [orcl.testkit.proto :as proto]
            [clojure.test :as test]
    #?(:clj
            [capacitor.core :as capacitor])
    #?(:clj
            [criterium.core :as criterium]))
  (:refer-clojure :exclude [compile])
  #?(:clj
     (:import [java.net URI]))
  #?(:clj
     (:gen-class)))

(defonce compiler (atom nil))

(defn normalize-desc [x]
  (cond
    (map? x) (update x :coeffects (fn [coeffects] (map #(update % :expectations normalize-desc) coeffects)))
    (vector? x) {:expectations (mapv (fn [v] {:type :basic :value v}) x)}
    (set? x) {:expectations [{:type :permutable :values x}]}
    :else {:expectations [{:type :basic :value x}]}))

(defn normalize-values-spec [spec]
  (if (or (vector? spec) (set? spec) (map? spec))
    spec
    [spec]))

(defn normalize-spec [spec]
  (if (map? spec)
    (update spec :values normalize-values-spec)
    {:values (normalize-values-spec spec)}))

(defn coeffect-id-by-definition [pending-coeffects definition]
  (some (fn [[id d]] (when (= d definition) id)) pending-coeffects))

(defn check-res [snapshot spec pending-coeffects]
  (let [res  (compiler/values snapshot)
        spec (normalize-spec spec)]
    (let [failed (cond
                   (set? (:values spec))
                   (or (not= (count (:values spec)) (count res))
                       (not= (:values spec) (set res)))

                   (and (map? (:values spec)) (::tests/one-of (:values spec)))
                   (or (not= 1 (count res))
                       (not (contains? (::tests/one-of (:values spec)) (first res))))

                   :else (not= (:values spec) res))]
      (when failed
        (throw (ex-info "Unexpected values" {:expected (:values spec) :actual res}))))
    (let [expected-killed (map (partial coeffect-id-by-definition pending-coeffects) (:killed-coeffects spec))]
      (when (not= (set expected-killed) (set (compiler/killed-coeffects snapshot)))
        (throw (ex-info "Unexpected killed-coeffects" {:expected expected-killed :actual (compiler/killed-coeffects snapshot)}))))))

(defn run-and-check [compiled [_ run-spec & unblock-specs]]
  (loop [snapshot          (compiler/eval compiled)
         spec              {:values run-spec}
         [[coeffect-definition realized-value next-spec] & unblock-specs] unblock-specs
         pending-coeffects {}]
    (check-res snapshot spec pending-coeffects)
    (let [pending-coeffects (apply dissoc (merge pending-coeffects (into {} (compiler/coeffects snapshot)))
                                   (compiler/killed-coeffects snapshot))]
      (cond
        (and coeffect-definition (empty? pending-coeffects))
        (throw (ex-info "Expected coeffect" {:coeffect coeffect-definition}))

        coeffect-definition
        (let [coeffect-id (coeffect-id-by-definition pending-coeffects coeffect-definition)]
          (recur (compiler/unblock snapshot coeffect-id realized-value)
                 next-spec
                 unblock-specs
                 (dissoc pending-coeffects coeffect-id)))

        (seq pending-coeffects)
        (throw (ex-info "Pending coeffects" {:pending-coeffects pending-coeffects}))))))

(defn compile [compiler test]
  (let [[program] test]
    (proto/compile compiler program)))

(defn run-test [compiler t]
  (try
    (run-and-check (compile compiler t) t)
    (catch clojure.lang.ExceptionInfo e
      (prn e) (throw e)))
  true)

(test/deftest main
  (doseq [[suite tests] tests/tests]
    (doseq [[program :as t] tests]
      (prn program)
      (test/is (run-test @compiler t)))))

(defn required-env [n]
  (or (System/getenv n)
      (do
        (prn (format "Required ENV variable %s" n))
        (System/exit 1))))

(comment
  (orcl.testkit.core/benchmarks (orcl.naive.tests-runner/compiler)
                                {:tags           ["backend=naive2"]
                                 :influx-connect "http://influxdb.service.consul:8086"
                                 :influx-db      "jmh_hub_prepor"
                                 :influx-user    ""
                                 :influx-pass    ""}))

#?(:clj
   (defn benchmarks [compiler {:keys [tags influx-connect influx-user influx-pass influx-db]}]
     (let [addr    (URI. influx-connect)
           influx  (capacitor/make-client {:host     (.getHost addr)
                                           :scheme   (.getScheme addr)
                                           :port     (.getPort addr)
                                           :username influx-user
                                           :password influx-pass
                                           :db       influx-db})
           results (doall (for [[suite tests] (select-keys tests/tests [:arithmetic :defs])
                                :let [_        (prn "Benchmarking" suite)
                                      compiled (map (partial compile compiler) tests)
                                      res      (criterium/quick-benchmark
                                                 (doseq [[compiled t] (map vector compiled tests)]

                                                   (run-and-check compiled t))
                                                 {})
                                      [mean] (:mean res)]]
                            [suite mean]))
           now     (System/currentTimeMillis)
           tags'   (into {} (map (fn [t] (let [[_ tag value] (re-find #"(.+)=(.+)" t)]
                                           (when-not tag
                                             (prn (format "Bad formatted tag %s" t))
                                             (System/exit 1))
                                           [tag value]))
                                 tags))]
       (capacitor/write-points influx
                               (for [[suite mean] results]
                                 {:measurement (name suite)
                                  :tags        tags'
                                  :fields      {"value" mean}
                                  :timestamp   now})))))

#?(:clj
   (defn -main [compiler-name & [mode & args]]
     (let [compiler-sym (symbol compiler-name)
           _            (require (symbol (namespace compiler-sym)))]
       (reset! compiler ((resolve compiler-sym)))
       (case mode
         "benchmarks" (benchmarks compiler {:tags           args
                                            :influx-connect (required-env "INFLUX_CONNECT")
                                            :influx-user    (System/getenv "INFLUX_USER")
                                            :influx-pass    (System/getenv "INFLUX_PASS")
                                            :influx-db      (required-env "INFLUX_DB")})
         (test/run-tests 'orcl.testkit.core)))))