(ns orcl.testkit.core
  (:require [orcl.testkit.tests :as tests])
  (:gen-class)
  (:import (java.util ServiceLoader)
           (orcl.testkit TestOrcFactory)))

(defn normalize-desc [x]
  (cond
    (map? x) x
    (vector? x) {:expectations (mapv (fn [v] {:type :basic :value v}) x)}
    (set? x) {:expectations [{:type :permutable :values x}]}
    :else {:expectations [{:type :basic :value x}]}))

(def tests (select-keys tests/tests [:arithmetic]))

(defn run-and-check [program test-desc]
  (let [impl (first (ServiceLoader/load TestOrcFactory))
        values
             (loop [res (.run impl program)]
               (let [coeffects (seq (.getCoeffects res))]
                 (if coeffects
                   (let [coeffect   (first coeffects)
                         id         (.getId coeffect)
                         definition (.getDefinition coeffect)
                         v          (or (get (:blocks test-desc) definition)
                                        (throw (ex-info "Unexpected coeffect" {:definition definition})))]
                     (recur (.unblock res id v)))
                   (.getValues res))))]
    (loop [[e & expectations] (:expectations test-desc) values values]
      (cond
        (and (nil? e) (empty? values)) :ok
        (and e (empty? values)) (throw (ex-info "Value expected" {:expectation e}))
        (and (nil? e) (seq values)) (throw (ex-info "Unexpected value" {:values values}))
        :else (case (:type e)
                :basic (if (= (:value e) (first values))
                         (recur expectations (rest values))
                         (throw (ex-info "Unexpected value" {:value       (first values)
                                                             :expectation e})))
                :permutable (if (contains? (:values e) (first values))
                              (if (= 1 (count (:values e)))
                                (recur expectations (rest values))
                                (recur (cons (update e :values disj (first values)) expectations) (rest values)))
                              (throw (ex-info "Unexpected value" {:value       (first values)
                                                                  :expectation e}))))))))

(defn -main [& args]
  (doseq [[suite tests] tests]
    (prn (str suite))
    (doseq [[i [t desc]] (map-indexed vector tests)]
      (prn "---T" t)
      (try
        (run-and-check t (normalize-desc desc))
        (catch clojure.lang.ExceptionInfo e
          (prn e) (throw e))))))