(ns orcl.compiler
  (:require [orcl.fs :as fs]
            [clojure.string :as str]
            [orcl.utils :as utils]
            [orcl.parser :as parser]
            [orcl.analyzer :as analyzer]
            [clojure.set :as set])
  (:refer-clojure :exclude [compile resolve eval]))

(defprotocol Backend
  (prelude [this])
  (analyzer-options [this])
  (compile [this ast] [this ast dependencies])
  (compile-namespace [this ast] [this ast dependencies]))

(defprotocol Program
  (eval [this] [this dependencies]))

(defprotocol Snapshot
  (values [this])
  (killed-coeffects [this])
  (coeffects [this])
  (unblock [this coeffect value]))

(defprotocol Resolver
  (resolve [this ns on-source]))

(defn ns->path [ns] (str (str/replace ns #"\." "/") ".orc"))

(defn basic-resolver [fs]
  (reify Resolver
    (resolve [this ns on-source]
      (if-let [source (fs/read-file fs (ns->path ns))]
        (on-source source)
        (utils/todo-exception)))))

(defn in-memory-cache-resolver [fs]
  (let [basic (basic-resolver fs)
        cache (atom {})]
    (reify Resolver
      (resolve [this ns on-source]
        (if-let [v (get @cache ns)]
          v
          (let [compiled (resolve basic ns on-source)]
            (swap! cache assoc ns compiled)
            compiled))))))

(defn parse-ns [resolver fs ns]
  (resolve resolver ns #(parser/parse % fs)))

(def stdlib-symbols
  ["abs" "signum" "min" "max"
   "curry" "curry3" "uncarry" "uncarry3" "flip" "constant" "defer" "defer2" "ignore" "ignore2" "compose"
   "while" "repeat" "fork" "forkMap" "seq" "seqMap" "join" "joinMap" "alt" "altMap" "por" "pand"
   "each" "map" "reverse" "filter" "head" "tail" "init" "last" "empty" "index" "append" "foldl" "foldl1"
   "foldr" "foldr1" "afold" "zipWith" "zip" "unzip" "concat" "length" "take" "drop" "member" "merge"
   "mergeBy" "sort" "sortBy" "mergeUnique" "mergeUniqueBy" "sortUnique" "sortUniqueBy" "group" "groupBy" "rangeBy" "range"
   "any" "all" "sum" "product" "and" "or" "minimum" "maximum"])

(defn inject-stdlib [ast fs]
  (let [ns      "com.360dialog.orcl.prelude"]
    (-> ast
        (update :dependencies conj ns)
        (assoc :body {:node  :declarations
                      :decls [{:type      :refer
                               :namespace ns
                               :symbols   stdlib-symbols}]
                      :expr  (:body ast)}))))

(defn run
  [backend source & [{:keys [fs resolver stdlib?]
                      :or   {fs       (fs/in-memory-file-system {})
                             resolver (basic-resolver fs)
                             stdlib true}}]]
  (let [options      (analyzer-options backend)
        lib          (prelude backend)
        dependencies (atom {})]
    ;; TODO cyclic dependencies detection
    (letfn [(on-source-clb [source]
              (let [parsed (parser/parse-namespace source fs)]
                (compile-dependencies (:dependencies parsed))
                (compile-namespace backend (analyzer/analyze-namespace parsed lib options) @dependencies)))
            (compile-dependencies [namespaces]
              (doseq [ns namespaces]
                (swap! dependencies assoc ns (resolve resolver ns on-source-clb))))]
      (let [parsed   (-> (parser/parse source fs)
                         (cond-> stdlib? (inject-stdlib fs)))
            _        (compile-dependencies (:dependencies parsed))
            analyzed (analyzer/analyze parsed lib options)]
        (eval (compile backend analyzed @dependencies) @dependencies)))))