(ns orcl.compiler
  (:require [orcl.fs :as fs]
            [clojure.string :as str]
            [orcl.utils :as utils]
            [orcl.parser :as parser]
            [orcl.analyzer :as analyzer]
            [clojure.set :as set])
  (:refer-clojure :exclude [compile resolve eval]))

(defprotocol Backend
  (prelude [this])
  (analyzer-options [this])
  (compile [this ast] [this ast dependencies])
  (compile-namespace [this ast] [this ast dependencies]))

(defprotocol Program
  (eval [this] [this dependencies]))

(defprotocol Snapshot
  (values [this])
  (killed-coeffects [this])
  (coeffects [this])
  (unblock [this coeffect value]))

(defprotocol Resolver
  (resolve [this ns on-source]))

(defn basic-resolver [fs]
  (reify Resolver
    (resolve [this ns on-source]
      (let [path (str (str/replace ns #"\." "/") ".orc")]
        (if-let [source (fs/read-file fs path)]
          (on-source source)
          (utils/todo-exception))))))

(defn in-memory-cache-resolver [fs]
  (let [basic (basic-resolver fs)
        cache (atom {})]
    (reify Resolver
      (resolve [this ns on-source]
        (if-let [v (get @cache ns)]
          v
          (let [compiled (resolve basic ns on-source)]
            (swap! cache assoc ns compiled)
            compiled))))))

(defn parse-ns [resolver fs ns]
  (resolve resolver ns #(parser/parse % fs)))

(defn run
  [backend source & [{:keys [fs resolver] :or {fs       (fs/in-memory-file-system {})
                                               resolver (basic-resolver fs)}}]]
  (let [options      (analyzer-options backend)
        lib          (prelude backend)
        dependencies (atom {})]
    ;; TODO cyclic dependencies detection
    (letfn [(on-source-clb [source]
              (let [parsed (parser/parse-namespace source fs)]
                (compile-dependencies (:dependencies parsed))
                (compile-namespace backend (analyzer/analyze-namespace parsed lib options) @dependencies)))
            (compile-dependencies [namespaces]
              (doseq [ns namespaces]
                (swap! dependencies assoc ns (resolve resolver ns on-source-clb))))]
      (let [parsed   (parser/parse source fs)
            _        (compile-dependencies (:dependencies parsed))
            analyzed (analyzer/analyze parsed lib options)]
        (eval (compile backend analyzed @dependencies) @dependencies)))))