(ns orcl.naive.lib
  (:require [orcl.naive.impl :as impl]
            [orcl.naive.vars :as vars]
    #?(:clj
            [orcl.naive.macro :as macro :refer [defsite]]))
  #?(:cljs (:require-macros
             [orcl.naive.macro :as macro :refer [defsite]])))

(defsite Let
         {:type :let}
         (impl/basic-site (fn [& args]
                            (case (count args)
                              0 :signal
                              1 (first args)
                              (vec args)))))

(defsite Ift {:params [{:type :boolean}]
              :return {:type :signal}}
         (impl/basic-site (fn [x] (if x :signal ::impl/halt))))

(defsite Iff {:params [{:type :boolean}]
              :return {:type :signal}}
         (impl/basic-site (fn [x] (if-not x :signal ::impl/halt))))

(defsite ^{:site "+"} plus
         {:type         :overload
          :alternatives [{:type   :fun
                          :params [{:type :integer}
                                   {:type :integer}]
                          :return {:type :integer}}
                         {:type   :fun
                          :params [{:type :number}
                                   {:type :number}]
                          :return {:type :number}}
                         {:type   :fun
                          :params [{:type :top}
                                   {:type :string}]
                          :return {:type :string}}
                         {:type   :fun
                          :params [{:type :string}
                                   {:type :top}]
                          :return {:type :string}}]}
         (impl/basic-site
           (fn [a b] (cond
                       (or (string? a) (string? b)) (str a b)
                       (map? a) (merge a b)
                       :else (+ a b)))))
(defsite ^{:site "-"} minus
         {:type         :overload
          :alternatives [{:type   :fun
                          :params [{:type :integer}
                                   {:type :integer}]
                          :return {:type :integer}}
                         {:type   :fun
                          :params [{:type :number}
                                   {:type :number}]
                          :return {:type :number}}]}
         (impl/basic-site -))

(defsite ^{:site "UMinus"} minus
         {:type         :overload
          :alternatives [{:type   :fun
                          :params [{:type :integer}]
                          :return {:type :integer}}
                         {:type   :fun
                          :params [{:type :number}]
                          :return {:type :number}}]}
         (impl/basic-site -))

(defsite ^{:site "0-"} negative
         {:params [{:type :number}]
          :return {:type :number}}
         (impl/basic-site -))
(defsite ^{:site "*"} mult
         {:params [{:type :number} {:type :number}]
          :return {:type :number}}
         (impl/basic-site *))
(defsite ^{:site "**"} pow
         {:params [{:type :number} {:type :number}]
          :return {:type :number}}
         (impl/basic-site (fn [base pow] #?(:clj  (Math/pow base pow)
                                            :cljs (.pow js/Math base pow)))))
(defsite ^{:site "/"} div
         {:params [{:type :number} {:type :number}]
          :return {:type :number}}
         (impl/basic-site /))
(defsite ^{:site "%"} rem-op
         {:params [{:type :number} {:type :number}]
          :return {:type :number}}
         (impl/basic-site rem))
(defsite ^{:site "<:"} less
         {:params [{:type :top} {:type :top}]
          :return {:type :boolean}}
         (impl/basic-site <))
(defsite ^{:site "<="} less-or-eq
         {:params [{:type :top} {:type :top}]
          :return {:type :boolean}}
         (impl/basic-site <=))
(defsite ^{:site ":>"} greater
         {:params [{:type :top} {:type :top}]
          :return {:type :boolean}}
         (impl/basic-site >))
(defsite ^{:site ">="} greater-or-eq
         {:params [{:type :top} {:type :top}]
          :return {:type :boolean}}
         (impl/basic-site >=))
(defsite ^{:site "="} eq
         {:params [{:type :top} {:type :top}]
          :return {:type :boolean}}
         (impl/basic-site =))
(defsite ^{:site "/="} not-eq
         {:params [{:type :top} {:type :top}]
          :return {:type :boolean}}
         (impl/basic-site not=))
(defsite ^{:site "~"} not-op
         {:params [{:type :boolean}]
          :return {:type :boolean}}
         (impl/basic-site not))
(defsite ^{:site "&&"} and-op
         {:params [{:type :boolean} {:type :boolean}]
          :return {:type :boolean}}
         (impl/basic-site (fn [a b] (and a b))))
(defsite ^{:site "||"} or-op
         {:params [{:type :boolean} {:type :boolean}]
          :return {:type :boolean}}
         (impl/basic-site (fn [a b] (or a b))))
(defsite ^{:site ":"} cons-op
         {:type-params ["T"]
          :params      [{:type :var :name "T"}
                        {:type :application :name "List"
                         :args [{:type :var :name "T"}]}]
          :return      {:type :application
                        :name "List"
                        :args [{:type :var :name "T"}]}}
         (impl/basic-site cons))

;;  All father stuff should be implemented in stdlib

;(defsite abs (sfn [x] (if (< x 0) (- x) x)))
;(defsite signum (sfn [x] (cond
;                           (< x 0) -1
;                           (= x 0) 0
;                           :else 1)))
;(defsite ^{:site "min"} min-op (sfn [a b] (min a b)))
;(defsite ^{:site "max"} max-op (sfn [a b] (max a b)))
(defsite floor
         {:params [{:type :number}]
          :return {:type :integer}}
         (impl/basic-site (fn [x] #?(:clj (long (Math/floor x)) :cljs (.floor js/Math x)))))
(defsite ceil
         {:params [{:type :number}]
          :return {:type :integer}}
         (impl/basic-site (fn [x] #?(:clj (long (Math/ceil x)) :cljs (.ceil js/Math x)))))
(defsite sqrt
         {:params [{:type :number}]
          :return {:type :integer}}
         (impl/basic-site (fn [x] (if (< x 0)
                                    ::impl/halt
                                    #?(:clj (Math/sqrt x) :cljs (.sqrt js/Math x))))))


(defn coeffect-id []
  #?(:clj  (str (java.util.UUID/randomUUID))
     :cljs (str (random-uuid))))

(defsite Coeffect
         {:params [{:type :top}]
          :return {:type :top}}
         (fn [token [definition]]
           (swap! impl/*coeffects* assoc (coeffect-id) {:definition definition
                                                        :token      token})))

(defsite Println
         {:params [{:type :top}]
          :return {:type :signal}}
         (impl/basic-site (fn [x] (prn x) :signal)))
(defsite ^{:site "Error"} error-op
         {:params [{:type :top}]
          :return {:type :bot}}
         (impl/basic-site (fn [x] (prn "Error" x) ::impl/halt)))

(defsite Cell
         {:type-params ["T"]
          :return      {:type :application
                        :name "Cell"
                        :args [{:type :var :name "T"}]}}
         (impl/basic-site
           (fn []
             (let [value       (atom ::empty)
                   subscribers (atom [])]
               {"read"  (reify impl/Site
                          (site-call [_ token _]
                            (if (= ::empty @value)
                              (swap! subscribers conj token)
                              (impl/publish-and-halt token @value))))
                "readD" (reify impl/Site
                          (site-call [_ token _]
                            (if (= ::empty @value)
                              (impl/halt token)
                              (impl/publish-and-halt token @value))))
                "write" (reify impl/Site
                          (site-call [_ token [v]]
                            (if (= ::empty @value)
                              (do (reset! value v)
                                  (doseq [t @subscribers]
                                    (impl/schedule t))
                                  (impl/publish-and-halt token :signal))
                              (impl/halt token))))}))))

(defsite Ref
         {:type-params ["T"]
          :return      {:type :application
                        :name "Ref"
                        :args [{:type :var :name "T"}]}}
         (impl/basic-site
           (fn [init-value]
             (let [value (atom init-value)]
               {"read"  (reify impl/Site
                          (site-call [_ token _]
                            (impl/publish-and-halt token @value)))
                "readD" (reify impl/Site
                          (site-call [_ token _]
                            (impl/publish-and-halt token @value)))
                "write" (reify impl/Site
                          (site-call [_ token [v]]
                            (reset! value v)
                            (impl/publish-and-halt token :signal)))}))))

(defsite Channel
         {:type-params ["T"]
          :return      {:type :application
                        :name "Channel"
                        :args [{:type :var :name "T"}]}}
         (impl/basic-site
           (fn []
             (let [values          (atom #?(:clj clojure.lang.PersistentQueue/EMPTY :cljs cljs.core.PersistentQueue/EMPTY))
                   waiters         (atom [])
                   empty-waiters   (atom [])
                   state           (atom :open)
                   read-from-queue (fn []
                                     (when-let [v (first @values)]
                                       (let [values' (swap! values pop)]
                                         (when (empty? values')
                                           (doseq [w @empty-waiters]
                                             (impl/schedule w)))
                                         v)))]
               {"get"      (reify impl/Site
                             (site-call [_ token _]
                               (if-let [v (read-from-queue)]
                                 (impl/publish-and-halt token v)
                                 (if (= :open @state)
                                   (swap! waiters conj token)
                                   (impl/halt token)))))
                "getD"     (reify impl/Site
                             (site-call [_ token _]
                               (if-let [v (read-from-queue)]
                                 (impl/publish-and-halt token v)
                                 (impl/halt token))))
                "put"      (reify impl/Site
                             (site-call [_ token [v]]
                               (if (= :open @state)
                                 (do
                                   (swap! values conj v)
                                   (doseq [w @waiters]
                                     (impl/schedule w))
                                   (reset! waiters [])
                                   (impl/publish-and-halt token :signal))
                                 (impl/halt token))))
                "close"    (reify impl/Site
                             (site-call [_ token _]
                               (reset! state :closed)
                               (doseq [w @waiters]
                                 (impl/schedule w))
                               (reset! waiters [])
                               (if (empty? @values)
                                 (impl/publish-and-halt token :signal)
                                 (swap! empty-waiters conj token))))
                "closeD"   (reify impl/Site
                             (site-call [_ token _]
                               (reset! state :closed)
                               (doseq [w @waiters]
                                 (impl/schedule w))
                               (reset! waiters [])
                               (impl/publish-and-halt token :signal)))
                "isClosed" (reify impl/Site
                             (site-call [_ token _]
                               (impl/publish-and-halt token (= :closed @state))))
                "getAll"   (reify impl/Site
                             (site-call [_ token _]
                               (loop [acc []]
                                 (if-let [v (read-from-queue)]
                                   (recur (conj acc v))
                                   (impl/publish-and-halt token acc)))))}))))


(defsite _MakeTuple {:type :tuple-constructor}
         (impl/basic-site (fn [& args] (vec args))))

(defsite _MakeRecord {:type :record-constructor}
         (impl/basic-site hash-map))

(defsite _TupleArityChecker {:type :tuple-arity-checker}
         (impl/basic-site (fn [t n] (if (= (count t) n) t ::impl/halt))))

(defsite _Nil {:type-params ["T"]
               :params      []
               :return      {:type :application
                             :name "List"
                             :args [{:type :var :name "T"}]}}
         (impl/basic-site (fn [] ())))

(defsite _First {:type-params ["T"]
               :params      [{:type :application
                              :name "List"
                              :args [{:type :var :name "T"}]}]
               :return      {:type :var :name "T"}}
         (impl/basic-site (fn [l] (if (empty? l)
                                    ::impl/halt
                                    (first l)))))

(defsite _Rest {:type-params ["T"]
                 :params      [{:type :application
                                :name "List"
                                :args [{:type :var :name "T"}]}]
                 :return      {:type :application
                               :name "List"
                               :args [{:type :var :name "T"}]}}
         (impl/basic-site (fn [l] (if (empty? l)
                                         ::impl/halt
                                         (rest l)))))

(defsite _FieldAccess {} (impl/basic-site (fn [m k] (if (contains? m k) (get m k) ::impl/halt))))

(defsite _WrapSome {:type-params ["T"]
                    :params      [{:type :var :name "T"}]
                    :return      {:type :application
                                  :name "Option"
                                  :args [{:type :var :name "T"}]}}
         (impl/basic-site (fn [v] {::some v})))
(defsite _UnwrapSome {:type-params ["T"]
                      :params      [{:type :application
                                     :name "Option"
                                     :args [{:type :var :name "T"}]}]
                      :return      {:type :var :name "T"}}
         (impl/basic-site (fn [v]
                            (if (and (map? v) (contains? v ::some))
                              (::some v)
                              ::impl/halt))))

(defsite _None {:type-params ["T"]
                :params      [{:type :var :name "T"}]
                :return      {:type :application
                              :name "Option"
                              :args [{:type :var :name "T"}]}}
         (impl/basic-site (fn [] ::none)))

(defsite _IsNone {:type-params ["T"]
                  :params      [{:type :application
                                 :name "Option"
                                 :args [{:type :var :name "T"}]}]
                  :return      {:type :boolean}}
         (impl/basic-site (fn [v] (if (= ::none v)
                                    ::impl/signal
                                    ::impl/halt))))

(defsite _Unapply {:type :unapply}
         (impl/basic-site (fn [[tag & xs] expected]
                            (if (= expected tag)
                              (vec xs)
                              ::impl/halt))))

(def prelude
  (into {} (for [[x {:keys [T]}] @vars/prelude]
             [x {:type       :site :source {:type :prelude
                                            :T    T}
                 :definition x}])))