(ns orcl.naive
  (:require [orcl.naive.impl :as impl]
            [orcl.naive.vars :as vars]
            [orcl.utils :as utils]
            [orcl.compiler :as compiler]
            [orcl.naive.lib :as lib])
  (:refer-clojure :exclude [compile]))

(defn prepare-primitive [ast]
  (if (= :var (:node ast))
    (let [s (:source ast)]
      (case (:type s)
        :site {:node  :const
               :value (impl/make-site (:type (:source s)) (:definition s))}
        :refer {:node      :refer
                :namespace (:namespace s)
                :symbol    (:var ast)}
        {:node :var
         :var  (:var ast)}))
    ast))

(defn prelude-const [definition]
  {:node  :const
   :value (impl/make-site :prelude definition)})

(defn pattern-binding [p]
  (case (:type p)
    :wildcard "_"
    :var (:var p)))

(declare compile)

(defn compile-def [def]
  {:name      (:name def)
   :free-vars (:free-vars def)
   :params    (map pattern-binding (:params (first (:instances def))))
   :body      (compile (:body (first (:instances def))))})

(defn compile [ast]
  (case (:node ast)
    (:sequential :pruning) {:node    (:node ast)
                            :binding (pattern-binding (:pattern ast))
                            :left    (compile (:left ast))
                            :right   (compile (:right ast))}
    (:otherwise :parallel) {:node  (:node ast)
                            :left  (compile (:left ast))
                            :right (compile (:right ast))}
    :defs-group (assoc ast
                  :expr (compile (:expr ast))
                  :defs (for [mr-group (:defs ast)
                              d        mr-group]
                          (compile-def d)))
    :call {:node   :call
           :target (prepare-primitive (:target ast))
           :args   (map prepare-primitive (:args ast))}
    :field-access {:node   :call
                   :target (prelude-const "_FieldAccess")
                   :args   [(prepare-primitive (:target ast)) {:node :const :value (:field ast)}]}
    (:stop :const) ast
    (:declare-types :has-type :refer) (compile (:expr ast))
    :var {:node   :call
          :target (prelude-const "Let")
          :args   [(prepare-primitive ast)]}))

(defn make-snapshot [values prev-coeffects killed dependencies]
  (let [coeffects (map (fn [[k {:keys [definition]}]] [k definition]) @impl/*coeffects*)
        state     (into {} (concat (map (fn [[k {:keys [token]}]] [k token]) @impl/*coeffects*)
                                   (apply dissoc prev-coeffects killed)))]
    (reify compiler/Snapshot
      (values [_] @values)
      (coeffects [_] coeffects)
      (killed-coeffects [_] killed)
      (unblock [_ coeffect value]
        (when-let [t (get state coeffect)]
          (reset! values [])
          (binding [impl/*coeffects*        (atom {})
                    impl/*exectution-queue* (atom ())
                    impl/*dependencies*     dependencies]
            (impl/publish-and-halt t value)
            (let [killed (keep (fn [[k {:keys [stack]}]] (when-not (impl/alive? stack) k)) (dissoc state coeffect))]
              (doseq [k killed]
                (impl/halt (get state k)))
              (impl/execution-loop)
              (make-snapshot values (dissoc state coeffect) killed dependencies))))))))

(defn eval* [program dependencies]
  (let [values (atom [])]
    (binding [impl/*coeffects*        (atom {})
              impl/*exectution-queue* (atom ())
              impl/*dependencies*     dependencies]
      (impl/execution-loop program (impl/result-frame (fn [v] (swap! values conj v))))
      (make-snapshot values {} [] dependencies))))

(defn backend []
  (reify compiler/Backend
    (prelude [_] lib/prelude)
    (analyzer-options [_] {:deflate?         true
                           :patterns?        true
                           :clauses?         true
                           :conditional?     true
                           :data-structures? true
                           :typecheck?       true})
    (compile [this ast] (compiler/compile this ast {}))
    (compile [_ ast dependenices]
      (let [compiled (compile ast)]
        (reify compiler/Program
          (eval [this] (compiler/eval this {}))
          (eval [_ dependenices]
            ;(prn "---EV" compiled)
            (eval* compiled dependenices)))))
    (compile-namespace [this ast] (compiler/compile-namespace this ast {}))
    (compile-namespace [_ decls dependencies]
      (let [defs  (for [[_ d] decls]
                    (compile-def d))
            group {:env  {}
                   :defs defs}]
        (into {} (for [{:keys [name] :as d} defs]
                   [name (impl/make-closure group d)]))))))

#?(:cljs
   (defn define-custom [prefix module]
     (swap! impl/*dynamic-sites* assoc prefix module)))