(ns orcl.naive.lib
  (:require [orcl.naive.impl :as impl]
            [orcl.naive.vars :as vars]
    #?(:clj
            [orcl.naive.macro :as macro :refer [defsite]]))
  #?(:cljs (:require-macros
             [orcl.naive.macro :as macro :refer [defsite]])))

(defsite Let (impl/basic-site (fn [& args]
                                (case (count args)
                                  0 ::impl/signal
                                  1 (first args)
                                  (vec args)))))

(defsite Ift (impl/basic-site (fn [x] (if x ::impl/signal ::impl/halt))))

(defsite Iff (impl/basic-site (fn [x] (if-not x ::impl/signal ::impl/halt))))

(defsite ^{:site "+"} plus (impl/basic-site
                             (fn [a b] (cond
                                         (string? a) (str a b)
                                         (map? a) (merge a b)
                                         :else (+ a b)))))
(defsite ^{:site "-"} minus (impl/basic-site -))
(defsite ^{:site "0-"} negative (impl/basic-site -))
(defsite ^{:site "*"} mult (impl/basic-site *))
(defsite ^{:site "**"} pow (impl/basic-site (fn [base pow] #?(:clj  (Math/pow base pow)
                                                              :cljs (.pow js/Math base pow)))))
(defsite ^{:site "/"} div (impl/basic-site /))
(defsite ^{:site "%"} rem-op (impl/basic-site rem))
(defsite ^{:site "<:"} less (impl/basic-site <))
(defsite ^{:site "<="} less-or-eq (impl/basic-site <=))
(defsite ^{:site ":>"} greater (impl/basic-site >))
(defsite ^{:site ">="} greater-or-eq (impl/basic-site >=))
(defsite ^{:site "="} eq (impl/basic-site =))
(defsite ^{:site "/="} not-eq (impl/basic-site not=))
(defsite ^{:site "~"} not-op (impl/basic-site not))
(defsite ^{:site "&&"} and-op (impl/basic-site (fn [a b] (and a b))))
(defsite ^{:site "||"} or-op (impl/basic-site (fn [a b] (or a b))))
(defsite ^{:site ":"} cons-op (impl/basic-site cons))

;;  All father stuff should be implemented in stdlib

;(defsite abs (sfn [x] (if (< x 0) (- x) x)))
;(defsite signum (sfn [x] (cond
;                           (< x 0) -1
;                           (= x 0) 0
;                           :else 1)))
;(defsite ^{:site "min"} min-op (sfn [a b] (min a b)))
;(defsite ^{:site "max"} max-op (sfn [a b] (max a b)))
(defsite floor (impl/basic-site (fn [x] #?(:clj (long (Math/floor x)) :cljs (.floor js/Math x)))))
(defsite ceil (impl/basic-site (fn [x] #?(:clj (long (Math/ceil x)) :cljs (.ceil js/Math x)))))
(defsite sqrt (impl/basic-site (fn [x] (if (< x 0)
                                         ::impl/halt
                                         #?(:clj (Math/sqrt x) :cljs (.sqrt js/Math x))))))


(defn coeffect-id []
  #?(:clj  (str (java.util.UUID/randomUUID))
     :cljs (str (random-uuid))))

(defsite Coeffect
         (fn [token [definition]]
           (swap! impl/*coeffects* assoc (coeffect-id) {:definition definition
                                                        :stack      (:stack token)})))

(defsite Println (impl/basic-site (fn [x] (prn x) ::impl/signal)))
(defsite ^{:site "Error"} error-op (impl/basic-site (fn [x] (prn "Error" x) ::impl/halt)))

(defsite Cell
         (impl/basic-site
           (fn []
             (let [value       (atom ::empty)
                   subscribers (atom [])]
               {"read"  (reify impl/Site
                          (site-call [_ token _]
                            (if (= ::empty @value)
                              (do
                                (swap! subscribers conj token))
                              (impl/publish-and-halt token @value))))
                "readD" (reify impl/Site
                          (site-call [_ token _]
                            (if (= ::empty @value)
                              (impl/halt token)
                              (impl/publish-and-halt token @value))))
                "write" (reify impl/Site
                          (site-call [_ token [v]]
                            (if (= ::empty @value)
                              (do (reset! value v)
                                  (doseq [t @subscribers]
                                    (impl/schedule t))
                                  (impl/publish-and-halt token ::impl/signal))
                              (impl/halt token))))}))))

(defsite Ref
         (impl/basic-site
           (fn []
             (let [value       (atom ::empty)
                   subscribers (atom [])]
               {"read"  (reify impl/Site
                          (site-call [_ token _]
                            (if (= ::empty @value)
                              (do
                                (swap! subscribers conj token))
                              (impl/publish-and-halt token @value))))
                "readD" (reify impl/Site
                          (site-call [_ token _]
                            (if (= ::empty @value)
                              (impl/halt token)
                              (impl/publish-and-halt token @value))))
                "write" (reify impl/Site
                          (site-call [_ token [v]]
                            (reset! value v)
                            (doseq [t @subscribers]
                              (impl/schedule t))
                            (reset! subscribers [])
                            (impl/publish-and-halt token ::impl/signal)))}))))

(defsite Channel
         (impl/basic-site
           (fn []
             (let [values       (atom #?(:clj clojure.lang.PersistentQueue/EMPTY :cljs cljs.core.PersistentQueue/EMPTY))
                   waiters (atom [])
                   empty-waiters (atom [])
                   state (atom :open)
                   read-from-queue (fn []
                                     (when-let [v (first @values)]
                                       (let [values' (swap! values pop)]
                                         (when (empty? values')
                                           (doseq [w @empty-waiters]
                                             (impl/schedule w)))
                                         v)))]
               {"get"      (reify impl/Site
                             (site-call [_ token _]
                               (if-let [v (read-from-queue)]
                                 (impl/publish-and-halt token v)
                                 (if (= :open @state)
                                   (swap! waiters conj token)
                                   (impl/halt token)))))
                "getD"     (reify impl/Site
                             (site-call [_ token _]
                               (if-let [v (read-from-queue)]
                                 (impl/publish-and-halt token v)
                                 (impl/halt token))))
                "put"      (reify impl/Site
                             (site-call [_ token [v]]
                               (if (= :open @state)
                                 (do
                                   (swap! values conj v)
                                   (doseq [w @waiters]
                                     (impl/schedule w))
                                   (reset! waiters [])
                                   (impl/publish-and-halt token ::impl/signal))
                                 (impl/halt token))))
                "close"    (reify impl/Site
                             (site-call [_ token _]
                               (reset! state :closed)
                               (doseq [w @waiters]
                                 (impl/schedule w))
                               (reset! waiters [])
                               (if (empty? @values)
                                 (impl/publish-and-halt token ::impl/signal)
                                 (swap! empty-waiters conj token))))
                "closeD"   (reify impl/Site
                             (site-call [_ token _]
                               (reset! state :closed)
                               (doseq [w @waiters]
                                 (impl/schedule w))
                               (reset! waiters [])
                               (impl/publish-and-halt token ::impl/signal)))
                "isClosed" (reify impl/Site
                             (site-call [_ token _]
                               (impl/publish-and-halt token (= :closed @state))))
                "getAll"   (reify impl/Site
                             (site-call [_ token _]
                               (loop [acc []]
                                 (if-let [v (read-from-queue)]
                                   (recur (conj acc v))
                                   (impl/publish-and-halt token acc)))))}))))


(defsite _MakeTuple (impl/basic-site (fn [& args] (vec args))))

(defsite _MakeList (impl/basic-site (fn [& args]
                                      (apply list args))))

(defsite _FieldAccess (impl/basic-site (fn [m k] (if (contains? m k) (get m k) ::impl/halt))))

(defsite _MakeRecord
         (impl/basic-site (fn [keys & vs] (zipmap keys vs))))

(defn _PatternExtract* [p v]
  (letfn [(check-coll
            ([ps vs k] (check-coll ps vs {} k))
            ([[p & ps] [v & vs] res k]
             (if p
               (check p v (fn [m] (check-coll ps vs (merge res m) k)))
               (k res))))
          (check [p v k]
            (case (:type p)
              :var (k {(:var p) v})
              :wildcard (k {})
              :const (if (= v (:value p))
                       (k {})
                       ::impl/halt)
              (:tuple :list) (if (and (sequential? v) (= (count (:patterns p)) (count v)))
                               (check-coll (:patterns p) v k)
                               ::impl/halt)
              :record (if (map? v)
                        (check-coll (map #(get v (first %)) (:pairs p)) (map second (:pairs p)) k)
                        ::impl/halt)
              :cons (if (and (sequential? v) (first v))
                      (let [x (first v) xs (rest v)]
                        (check (:head p) x (fn [res] (check (:tail p) xs (fn [res'] (k (merge res res')))))))
                      ::impl/halt)
              :as (check (:pattern p) v (fn [res] (k (assoc res (:alias p) v))))))]
    (check p v identity)))

(defsite _PatternExtract (impl/basic-site _PatternExtract*))

(defsite _PatternGet (impl/basic-site (fn [v part] (get v part))))

(defsite _WrapSome (impl/basic-site (fn [v] {::some v})))
(defsite _UnwrapSome (impl/basic-site (fn [v]
                                        (if (and (map? v) (contains? v ::some))
                                          (::some v)
                                          ::impl/halt))))

(defsite _None (impl/basic-site (fn [] ::none)))

(defsite _IsNone (impl/basic-site (fn [v] (if (= ::none v)
                                            ::impl/signal
                                            ::impl/halt))))

(def prelude
  (into {} (for [x (keys @vars/prelude)] [x {:type :site :source {:type :prelude} :definition x}])))