(ns orcl.naive
  (:require [orcl.naive.impl :as impl]
            [orcl.naive.vars :as vars]
            [orcl.utils :as utils]
            [orcl.compiler :as compiler]
            [orcl.naive.lib :as lib]
            [clojure.string :as str])
  #?(:clj
     (:import [org.apache.commons.lang3.reflect MethodUtils]))
  (:refer-clojure :exclude [compile]))

#?(:cljs (def ^:dynamic *dynamic-sites* (atom {})))

;; Super inefficient. We need types to make it possible to compile to faster code
;; ?
(defn compile-import-site [s]
  (let [parts       (str/split s #"\.")
        class-name  (str/join "." (butlast parts))
        method-name (last parts)]
    #?(:clj
       (let [class (Class/forName class-name)]
         (impl/basic-site
           (fn [& args]
             (MethodUtils/invokeStaticMethod class method-name (into-array Object args)))))
       :cljs
       (let [obj (get @*dynamic-sites* class-name)]
         (impl/basic-site (get obj method-name))))))

(defn prepare-primitive [ast]
  (if (= :var (:node ast))
    (let [s (:source ast)]
      (case (:type s)
        :site {:node  :const
               :value (if (= :prelude (:type (:source s)))
                        (get @vars/prelude (:definition s))
                        (compile-import-site (:definition s)))}
        :refer {:node      :refer
                :namespace (:namespace s)
                :symbol    (:var ast)}
        {:node :var
         :var  (:var ast)}))
    ast))

(defn pattern-binding [p]
  (case (:type p)
    :wildcard "_"
    :var (:var p)))

(declare compile)

(defn compile-def [def]
  {:name   (:name def)
   :locals (:locals def)
   :params (map pattern-binding (:params (first (:instances def))))
   :body   (compile (:body (first (:instances def))))})

(defn compile [ast]
  (case (:node ast)
    (:sequential :pruning) {:node    (:node ast)
                            :binding (pattern-binding (:pattern ast))
                            :left    (compile (:left ast))
                            :right   (compile (:right ast))}
    (:otherwise :parallel) {:node  (:node ast)
                            :left  (compile (:left ast))
                            :right (compile (:right ast))}
    :defs-group (assoc ast
                  :expr (compile (:expr ast))
                  :defs (map compile-def (:defs ast)))
    :call {:node   :call
           :target (prepare-primitive (:target ast))
           :args   (map prepare-primitive (:args ast))}
    (:tuple :list) {:node   :call
                    :target {:node :const :value (case (:node ast) :tuple lib/make-tuple :list lib/make-list)}
                    :args   (map prepare-primitive (:values ast))}
    :record {:node   :call
             :target {:node :const :value lib/make-record}
             :args   (concat [{:node :const :value (map first (:pairs ast))}]
                             (map (comp prepare-primitive second) (:pairs ast)))}
    :field-access {:node   :call
                   :target {:node :const :value lib/field-access}
                   :args   [(prepare-primitive (:target ast)) {:node :const :value (:field ast)}]}
    (:stop :const) ast
    :var {:node   :call
          :target {:node :const :value lib/Let}
          :args   [(prepare-primitive ast)]}))

(defn make-snapshot [values prev-coeffects dependencies]
  (let [coeffects (map (fn [[k {:keys [definition]}]] [k definition]) @impl/*coeffects*)
        killed    (keep (fn [[k stack]] (when-not (impl/alive? stack) k)) prev-coeffects)
        state     (into {} (concat (map (fn [[k {:keys [stack]}]] [k stack]) @impl/*coeffects*)
                                   (apply dissoc prev-coeffects killed)))]
    (reify compiler/Snapshot
      (values [_] @values)
      (coeffects [_] coeffects)
      (killed-coeffects [_] killed)
      (unblock [_ coeffect value]
        (reset! values [])
        (binding [impl/*coeffects*        (atom {})
                  impl/*exectution-queue* (atom ())
                  impl/*dependencies*     dependencies]
          (let [stack (get state coeffect)]
            (impl/publish stack value)
            (impl/halt stack))
          (impl/execution-loop)
          (make-snapshot values (dissoc state coeffect) dependencies))))))

(defn eval* [program dependencies]
  (let [values (atom [])]
    (binding [impl/*coeffects*        (atom {})
              impl/*exectution-queue* (atom ())
              impl/*dependencies*     dependencies]
      (impl/execution-loop program (impl/result-frame (fn [v] (swap! values conj v))))
      (make-snapshot values {} dependencies))))

(defn backend []
  (reify compiler/Backend
    (prelude [_] lib/prelude)
    (analyzer-options [_] {:deflate?     true
                           :patterns?    true
                           :clauses?     true
                           :conditional? true})
    (compile [this ast] (compiler/compile this ast {}))
    (compile [_ ast dependenices]
      (let [compiled (compile ast)]
        (reify compiler/Program
          (eval [this] (compiler/eval this {}))
          (eval [_ dependenices]
            (eval* compiled dependenices)))))
    (compile-namespace [this ast] (compiler/compile-namespace this ast {}))
    (compile-namespace [_ decls dependencies]
      (let [defs (map compile-def decls)
            group {:env  {}
                   :defs defs}]
        (into {} (for [{:keys [name] :as d} defs]
                   [name (impl/make-closure group d)]))))))
