(ns orcl.naive.impl
  (:require [orcl.utils :as utils]
            [orcl.naive.vars :as vars])
  (:refer-clojure :exclude [realized?]))

(def ^:dynamic *exectution-queue* (atom ()))

(defn schedule [f] (swap! *exectution-queue* conj f))

(defprotocol IsRealized
  (realized? [this]))

(defprotocol Pending
  (realize [this v])
  (pending-bindings [this pattern]))

(defprotocol PendingBinding
  (value [this])
  (subscribe [this f ch]))

(defprotocol Channel
  (close [this])
  (open? [this] value)
  (write [this value]))

(defn pending? [x] (and (satisfies? PendingBinding x) (not (realized? x))))
(defn realized! [x]
  (if (satisfies? PendingBinding x)
    (value x)
    x))

(defn pattern-extractors [p]
  (case (:type p)
    :var [identity]
    (:wildcard :const) []
    (:list :tuple) (mapcat (fn [p i] (map (fn [f] (fn [v] (f (nth v i)))) (pattern-extractors p))) (:patterns p) (range))
    :record (mapcat (fn [[k p]] (map (fn [f] (fn [v] (f (get v k)))) (pattern-extractors p))) (:pairs p))
    :cons (concat (map (fn [f] (fn [v] (f (first v)))) (pattern-extractors (:head p)))
                  (map (fn [f] (fn [v] (f (rest v)))) (pattern-extractors (:tail p))))
    :as (cons identity (pattern-extractors (:pattern p)))))

(defn extract-pattern [p v]
  (map #(% v) (pattern-extractors p)))

(defn check-pattern [p v]
  (case (:type p)
    (:wildcard :var) true
    :const (= v (:value p))
    (:list :tuple) (and (sequential? v) (= (count v) (count (:patterns p)))
                        (every? (fn [[p v]] (check-pattern p v)) (map vector (:patterns p) v)))
    :record (and (map? v)
                 (every? (fn [[k p]] (check-pattern p (get v k ::not-found))) (:pairs p)))
    :cons (and (sequential? v)
               (let [[x & xs] v]
                 (and (check-pattern (:head p) x) (check-pattern (:tail p) xs))))
    :as (check-pattern (:pattern p) v)))

(defn pending []
  (let [subscribers  (atom [])
        value        (atom ::empty)
        make-binding (fn [extractor]
                       (reify PendingBinding
                         (value [_]
                           (extractor @value))
                         (subscribe [_ f ch] (swap! subscribers conj [f ch]))
                         IsRealized
                         (realized? [_] (not= ::empty @value))))]
    (reify Pending
      (realize [_ v]
        (reset! value v)
        (doseq [[f ch] @subscribers]
          (trampoline f ch)))
      (pending-bindings [this pattern]
        (map make-binding (pattern-extractors pattern)))
      IsRealized
      (realized? [_] (not= ::empty @value)))))

(defn constant [v]
  (fn [ch]
    (write @ch v)
    (close @ch)))

(defn stop [ch]
  (close @ch))

(defn channel
  ([clb on-close] (channel nil clb on-close))
  ([parent clb on-close]
   (let [open (atom true)
         me (atom nil)]
     (reset! me (reify Channel
             (write [this v] (clb me v))
             (open? [_] (and @open (or (nil? parent) (open? @parent))))
             (close [_]
               (when @open
                 (reset! open false)
                 (on-close)))))
     me)))

(defn sequential [left right-f]
  (fn [ch]
    (let [opened   (atom 1)
          closed   (fn [] (when (zero? (swap! opened dec))
                            (close @ch)))
          right-ch (channel ch
                            (fn [_ v] (write @ch v))
                            (fn [] (closed)))

          left-ch  (channel ch (fn [_ v]
                                 (swap! opened inc)
                                 (schedule #(trampoline (right-f v) right-ch)))
                            (fn []
                              (closed)
                              (reset! right-ch @ch)))]
      (trampoline left left-ch))))

(defn pruning [pattern left-f right]
  (fn [ch]
    (let [pending (pending)]
      (trampoline right (channel ch (fn new-val [this v]
                                      (when (not (realized? pending))
                                        (close @this)
                                        (realize pending v)))
                                 (fn [])))
      ((left-f (pending-bindings pending pattern)) ch))))

(defn parallel [left right]
  (fn [ch]
    (let [open     (atom 2)
          closed   (fn [] (when (zero? (swap! open dec))
                            (close @ch)))
          left-ch  (atom nil)
          right-ch (channel ch (fn [_ v] (write @ch v)) (fn [] (reset! left-ch @ch) (closed)))
          _        (reset! left-ch @(channel ch (fn [_ v] (write @ch v)) (fn [] (reset! right-ch @ch) (closed))))]
      (schedule #(trampoline left left-ch))
      (schedule #(trampoline right right-ch)))))

(defn otherwise [left right]
  (fn [ch]
    (let [first-value? (atom false)]
      (left (channel ch (fn [this v]
                          (reset! first-value? true)
                          (reset! this @ch)
                          (write @ch v))
                     (fn [] (if @first-value?
                              (close @ch)
                              (right ch))))))))

(defn site [f]
  (fn site-call [& args]
    (fn [ch]
      (if-let [pending (some #(when (pending? %) %) args)]
        (subscribe pending (apply site-call args) ch)
        #(try
           (apply f ch (map realized! args))
           (catch #?(:clj Exception :cljs js/Error) e
             (close @ch)))))))

(defn basic-site [f]
  (site (fn [ch & args]
          (let [x (apply f args)]
            (when-not (= ::halt x)
              (write @ch x))
            (close @ch)))))

(defn strict-pattern? [p]
  (case (:type p)
    (:wildcard :var) false
    true))

(defn function [instances]
  (fn function-call [& args]
    (fn [ch]
      (loop [[instance & instances] instances]
        (if instance
          (if-let [pending (some (fn [[pattern arg]] (when (and (strict-pattern? pattern) (pending? arg))
                                                       arg))
                                 (map vector (:params instance) args))]
            (subscribe pending (apply function-call args) ch)
            (let [vals (map realized! args)]
              (if (every? (fn [[p v]] (check-pattern p (realized! v)))
                          (map vector (:params instance) vals))
                ((apply (:body instance) (map extract-pattern (:params instance) vals)) ch)
                (recur instances))))
          (close @ch))))))

(defn call [target args]
  (fn [ch]
    (if (pending? target)
      (subscribe target (call target args) ch)
      #((apply (realized! target) args) ch))))

(defn execution-loop []
  (loop []
    (when-let [queue (seq @*exectution-queue*)]
      (reset! *exectution-queue* ())
      (loop [[f & queue] queue]
        (when f
          (f)
          (recur queue)))
      (recur))))

(def ^:dynamic *coeffects* (atom {}))