(ns orcl.naive2.impl
  (:refer-clojure :exclude [eval]))

(def ^:dynamic *exectution-queue* (atom ()))

(defn schedule [f] (swap! *exectution-queue* conj f))

(defprotocol Frame
  (publish [this v])
  (alive? [this]))

(defn result-frame [clb]
  (reify Frame
    (publish [this v] (clb v))
    (alive? [this] true)))

(defn increment-instances [token]
  (when-let [otherwise (:otherwise token)]
    (swap! (:instances otherwise) inc)))

(defn sequential-frame [token]
  (let [{:keys [binding right]} (:node token)]
    (reify Frame
      (publish [this v]
        (increment-instances token)
        (schedule (-> token (assoc :node right) (assoc-in [:env binding] v))))
      (alive? [this] (alive? (:stack token))))))

(defn pruning-frame [token pending]
  (let [realized (atom false)]
    (reify Frame
      (publish [this v] (when-not @realized
                          (reset! (::pending pending) v)
                          (reset! realized true)
                          (doseq [[binding w] @(:waiters pending)]
                            (schedule (assoc-in w [:env binding] v)))))
      (alive? [this] (and (not @realized) (alive? (:stack token)))))))

(defn token [node stack]
  {:node      node
   :stack     stack
   :env       {}
   :otherwise nil})

(defn pending [] {::pending (atom ::empty)
                  :waiters  (atom [])})

(defn pending? [x] (and (map? x) (::pending x) (= ::empty @(::pending x))))

(defn realized! [x] (if (and (map? x) (::pending x))
                      @(::pending x)
                      x))

(defn subscribe [pending binding token]
  (swap! (:waiters pending) conj [binding token]))

(defn halt [token]
  (when-let [otherwise (:otherwise token)]
    (when (zero? (swap! (:instances otherwise) dec))
      (if @(:first-value? otherwise)
        (when-let [prev-otherwise (get-in otherwise [:token :otherwise])]
          (halt prev-otherwise))
        (schedule (assoc (:token otherwise) :node (get-in otherwise [:token :node :right])))))))

(defn make-otherwise [token]
  (let [first-value? (atom false)
        otherwise    {:token        token
                      :instances    (atom 1)
                      :first-value? first-value?}]
    (assoc token :node (get-in token [:node :left])
                 :otherwise otherwise
                 :stack (reify Frame
                          (publish [_ v]
                            (when-not @first-value?
                              (reset! first-value? true))
                            (publish (:stack token) v))
                          (alive? [_] (alive? (:stack token)))))))

(defn lookup [token x] (if (= :const (:node x))
                         (:value x)
                         (get-in token [:env (:var x)])))

(defn find-pending [token args]
  (some (fn [x]
          (when (= :var (:node x))
            (let [v (get-in token [:env (:var x)])]
              (when (pending? v)
                [(:var x) v]))))
        args))

(defn make-closure [group def]
  {:group group :def def})

(defn with-closure-group [env closure-group]
  (into env (map (fn [def] [(:name def) (make-closure closure-group def)]) (:defs closure-group))))

(defn function-call [token closure]
  (let [env (into {} (concat (:env (:group closure))
                             (map (fn [from to] [to (lookup token from)]) (:args (:node token)) (:params (:def closure)))))]
    (schedule (assoc token :node (:body (:def closure))
                           :env (with-closure-group env (:group closure))))))

(defn eval [token]
  (let [n (:node token)]
    (case (:node n)
      :stop (halt token)

      :const (do (publish (:stack token) (lookup token n))
                 (halt token))

      :call (let [target (case (:node (:target n))
                           :const (:value (:target n))
                           :var (get-in token [:env (:var (:target n))]))]
              (cond
                (pending? target) (subscribe target (:target n) token)
                (fn? (realized! target)) (if-let [[binding pending] (find-pending token (:args n))]
                                           (subscribe pending binding token)
                                           ((realized! target) token (map (comp realized! (partial lookup token)) (:args n))))
                :else (function-call token (realized! target))))

      :parallel (do
                  (increment-instances token)
                  (schedule (assoc token :node (:left n)))
                  (schedule (assoc token :node (:right n))))

      :sequential (schedule (assoc token :node (:left n)
                                         :stack (sequential-frame token)))

      :pruning (let [pending (pending)]
                 (increment-instances token)
                 (schedule (assoc token :node (:right n)
                                        :stack (pruning-frame token pending)))
                 (schedule (-> token
                               (assoc-in [:env (:binding n)] pending)
                               (assoc :node (:left n)))))

      :otherwise (schedule (make-otherwise token))

      :defs-group (let [closure-group {:env  (select-keys (:env token) (:locals n))
                                       :defs (:defs n)}]
                    (schedule (assoc token :node (:expr n)
                                           :env (with-closure-group (:env token) closure-group)))))))

(defn basic-site [f]
  (fn [token args]
    (let [res (apply f args)]
      (if (= ::halt res)
        (halt token)
        (do
          (publish (:stack token) res)
          (halt token))))))

(defn execution-loop
  ([node stack]
   (schedule (token node stack))
   (execution-loop))
  ([]
   (loop []
     (when-let [queue (seq @*exectution-queue*)]
       (reset! *exectution-queue* ())
       (loop [[token & queue] queue]
         (when token
           ;(prn "---EVAL" (:node token))
           (eval token)
           (recur queue)))
       (recur)))))

(def ^:dynamic *coeffects* (atom {}))