(ns orcl.naive2
  (:require [orcl.naive2.impl :as impl]
            [orcl.naive2.vars :as vars]
            [orcl.utils :as utils]
            [orcl.naive2.lib :as lib])
  (:refer-clojure :exclude [compile]))

(defn compile-import-site [s]
  (utils/todo-exception))

(defn prepare-primitive [ast]
  (if (= :var (:node ast))
    (let [s (:source ast)]
      (case (:type s)
        :site (if (= :prelude (:type (:source s)))
                {:node  :const
                 :value (get @vars/prelude (:definition s))}
                (compile-import-site s))
        {:node :var
         :var  (:var ast)}))
    ast))

(defn pattern-binding [p]
  (case (:type p)
    :wildcard "_"
    :var (:var p)))

(declare compile)

(defn compile-def [def]
  {:name   (:name def)
   :locals (:locals def)
   :params (map pattern-binding (:params (first (:instances def))))
   :body   (compile (:body (first (:instances def))))})

(defn compile [ast]
  (case (:node ast)
    (:sequential :pruning) {:node    (:node ast)
                            :binding (pattern-binding (:pattern ast))
                            :left    (compile (:left ast))
                            :right   (compile (:right ast))}
    (:otherwise :parallel) {:node  (:node ast)
                            :left  (compile (:left ast))
                            :right (compile (:right ast))}
    :defs-group (assoc ast
                  :expr (compile (:expr ast))
                  :defs (map compile-def (:defs ast)))
    :call {:node   :call
           :target (prepare-primitive (:target ast))
           :args   (map prepare-primitive (:args ast))}
    (:tuple :list) {:node   :call
                    :target {:node :const :value (case (:node ast) :tuple lib/make-tuple :list lib/make-list)}
                    :args   (map prepare-primitive (:values ast))}
    :record {:node   :call
             :target {:node :const :value lib/make-record}
             :args   (concat [{:node :const :value (map first (:pairs ast))}]
                             (map (comp prepare-primitive second) (:pairs ast)))}
    :field-access {:node   :call
                   :target {:node :const :value lib/field-access}
                   :args   [(prepare-primitive (:target ast)) {:node :const :value (:field ast)}]}
    (:stop :const) ast
    :var {:node   :call
          :target {:node :const :value lib/Let}
          :args   [(prepare-primitive ast)]}))

(defn make-res [values prev-coeffects]
  (let [killed (keep (fn [[k stack]] (when-not (impl/alive? stack) k)) prev-coeffects)]
    {:values           @values
     :coeffects        (map (fn [[k {:keys [definition]}]] [k definition]) @impl/*coeffects*)
     :killed-coeffects killed
     :state            {:values    values
                        :coeffects (into {} (concat (map (fn [[k {:keys [stack]}]] [k stack]) @impl/*coeffects*)
                                                    (apply dissoc prev-coeffects killed)))}}))

(defn run [program]
  (let [values (atom [])]
    (binding [impl/*coeffects*        (atom {})
              impl/*exectution-queue* (atom ())]
      (impl/execution-loop program (impl/result-frame (fn [v] (swap! values conj v))))
      (make-res values {}))))

(defn unblock [{:keys [values coeffects]} coeffect value]
  (reset! values [])
  (binding [impl/*coeffects*        (atom {})
            impl/*exectution-queue* (atom ())]
    (let [stack (get coeffects coeffect)]
      (impl/publish stack value)
      (impl/halt stack))
    (impl/execution-loop)
    (make-res values (dissoc coeffects coeffect))))
