(ns orcl.naive.functional-test
  (:require [clojure.test :refer [deftest testing is]]
            [orcl.parser :as parser]
            [orcl.analyzer :as analyzer]
            [orcl.naive :as naive]
            [orcl.naive.vars :as vars]
            [orcl.naive.lib :as lib]))

;; Full form:
;; {:expectations [{:type :permutable :values #{1 2 3}}]
;;  :blocks       {[:receive :event} :event-data}
;; Shorthands:
;; [a b] => {:expectations [{:type :basic :value a} {:type :basic :value b}]}
;; #{a b} => {:expectations [{:type :permutable :values #{a b}]}
;; any => {:expectations [{:type :basic :value any}]}

(def tests
  {:arithmetic
   [["1 + 2" 3]
    ["7 * 3" 21]
    ["30 / 6" 5]
    ["5 - 4" 1]
    ["(95 + 4 + 1) * 21" 2100]
    ["23 = 20 + 3" true]
    ["(7 >= 2 + 4)" true]
    ["15 % 4" 3]
    ["2**8" 256.0]
    ["5 / 2.0" 2.5]
    ["-(29)" -29]
    ;; TODO
    ;["(+)(4,5)" 9]
    ]

   :prelude
   [["true && (true || false)" true]

    ["\"All your \" + \"base are \" + \"belong to us\""
     "All your base are belong to us"]

    ["7 = 7" true]
    ["278 /= 290" true]
    ["6 <: 10" true]
    ["23 :> 900" false]
    ["8 <= 2" false]
    ["(7 >= 2 + 4)" true]
    ["98 >= 98" true]
    ["false && true" false]
    ["false || true" true]
    ["~true" false]
    ["86 = true" false]
    ["3 :> 2" true]
    ["6 <: 7" true]

    ["(7, 3, 2)" [[7 3 2]]]
    ["[8, 9, 10]" [[8 9 10]]]
    ["if false then 7 else 8" 8]

    ["val b = true
      Ift(b) >> \"true\" | Iff(b) >> \"false\"" "true"]


    ["val n = 5
      if n :> 0 then n-1 else n+1" 4]

    ["val x = 22
      x" 22]

    ["signal" :signal],
    ["stop" []],

    ["Let(2, 3, 4) >(x,y,z)> Let(x | z)" 2]
    ["abs(-4)", 4]
    ["signum(-19) | signum(0) | signum(20)", #{-1 0 1}]
    ["min(4,3)", 3]
    ["max(4,3)", 4]
    ["floor(3.2)", 3]
    ["ceil(3.2)", 4]
    ["sqrt(4)", 2.0]
    ["curry(min)(3)(2)" 2]
    ["curry3(Let)(3)(2)(1)" [[3 2 1]]]
    ["uncurry(curry(min))(4, 3)", 3]
    ["flip(Let)(1,2)", [[2, 1]]]
    ["constant(10)()", 10]
    ["defer(Let, 10)()", 10]
    ["defer2(Let, 10, 11)()", [[10, 11]]]
    ["compose(lambda (x) = x + 10, lambda (x) = x + 5)(5)", 20]
    ["while(
       lambda (n) = (n <= 5),
       lambda (n) = n+1
      )(0)", [0 1 2 3 4 5]]]

   :defs

   [["def f(a, b) = a + 2 * b
      f(1, 2)" 5]

    ["val (x,_,_) = (1,(2,2),[3,3,3])
      x" 1]

    ["def onetwosum(f) = f(1) + f(2)
      onetwosum( lambda(x) = x * 3 )" 9]

    ["def pow2(0) = 1
      def pow2(n) = 2 * pow2(n-1)

      pow2(8)" 256]

    ["def even(0) = true
      def odd(0) = false
      def even(n) = (if n :> 0 then n-1 else n+1) >i> odd(i)
      def odd(n) = (if n :> 0 then n-1 else n+1) >i> even(i)

      even(9)" false]

    ["def sumto(n) = if n <: 1 then 0 else n + sumto(n-1)
      sumto(10)" 55]

    ["def Sum(a) = ( lambda(b) = a+b )
      val f = Sum(3)
      f(4)" 7]

    ["def foo(x) =
       def bar(y) = x + y
       bar
     foo(10)(20)" 30]

    ["def Sum(Integer) :: lambda (Integer) :: Integer
      def Sum(a) = ( lambda(b) = a+b )
      val f = Sum(3)
      f(4)" 7]]

   :comments
   [["{- I
         am
         a
         multi-line
         comment
       -}

      {- {- I am a nested comment -} -}

      -- I am a single-line comment

      1 + {- I am hidden in an expression -} 1" 2]]

   :data-structures
   [["val empty = {.  .}
      val one = {. x = 1 .}
      val two = {. y = 3, z = true .}

      empty
      | one
      | (one.x)
      | (two.y)
      | (two.z)" #{{} {"x" 1} 1 3 true}]

    ["val a = {. x = 0 .} + {.  .}
      val b = {.  .} + {. y = 1 .}
      val c = {. x = 10 .} + {. x = 2 .}
      val d = {. x = 11, y = 4 .} + {. z = 5, x = 3 .}
      val e = {. x = {. y = 12 .}, z = 13 .} + {. x = {. y = 6, z = 7 .}, y = 8, z = 9 .}

      a.x | b.y | c.x | d.x | d.y | d.z | e.x.y | e.x.z | e.y | e.z"
     #{0 1 2 3 4 5 6 7 8 9}]

    ["[1,2,3] | [1]:4:[6] | 1:5:8:[] | [] | [7]"
     #{[1 2 3] [[1] 4 6] [1 5 8] () [7]}]]

   :patterns
   [["( (1,4) | (2,5) | (1,6) )  >(1,x)> x" #{4 6}]
    ["x <(1,x)<  ( (2,5) | (1,4) | (1,6) )" 4]
    ["( (1,2) | (1,2,3) | (5,6) | (5,6,7) ) >(x,y)> (y,x)" #{[2 1] [6 5]}]
    ["( [1,2,3] | [4,5] | [6] | [] ) >a:b:c> (a,b,c)" #{[1 2 [3]] [4 5 nil]}]
    ["[1,2,3,4] >a:t> t >b:c:u> (u,c,b,a)" [[[4] 3 2 1]]]
    ["(-1 >-1> \"ISuccess\" ; \"Fail\")
     |(-2.3 >-2.3> \"FSuccess\" ; \"Fail\")"
     #{"ISuccess" "FSuccess"}]
    ["( (1,(2,3)) | (4,(5,6)) ) >(a,(b,c) as d)> (a,b,c,d)"
     #{[1 2 3 [2 3]]
       [4 5 6 [5 6]]}]
    ["( (1,(2,3)) | (4,true) | (5,[6,7]) | (8,signal) ) >(x,_)> x" #{1 4 5 8}]]

   :stop-semantic
   [["((1 | stop) | stop); 3" 1]
    ["((1 >> stop) | stop); 3" 3]
    ["(1 << stop); 3" 1]
    ["(stop << 1); 3" 3]
    ["(1 | stop) >> 2" 2]]

   :combinators
   [["2 | 3" #{2 3}]
    ["2 + 3 >x> x" 5]
    ["y <y< (4 | 6)" 4]
    ["stop ; 5 + 9" 14]
    ["(Ift(b) >> 1 | Ift(~b) >> 0) <b< true" 1]
    ["1 + ( 2 | 3 )" 3]

    ["def picknum() = x <x< ( 1 | 2 )
      picknum()" 1]

    ["def pubNums(Integer) :: Integer
      def pubNums(n) = if(n :> 0) then (n | pubNums(n-1)) else stop
      pubNums(5) >x> x" [5 4 3 2 1]]

    ["1 >> (if true then 2 else 3)" 2]]

   :basic-blocks
   [["Coeffect(1) >x> x + 2"
     {:expectations [{:type :basic :value 3}]
      :blocks       {1 1}}]

    ["Coeffect(4) >x>
      (val y = 3
       x + y)"
     {:expectations [{:type :basic :value 7}]
      :blocks       {4 4}}]

    ["Coeffect(1) >(x, y)> x + y"
     {:expectations [{:type :basic :value 3}]
      :blocks       {1 [1 2]}}]]

   :core
   [["val c = Cell()
      c.read() | c.write(5)" 5]]
   })


(defn normalize-desc [x]
  (cond
    (map? x) x
    (vector? x) {:expectations (mapv (fn [v] {:type :basic :value v}) x)}
    (set? x) {:expectations [{:type :permutable :values x}]}
    :else {:expectations [{:type :basic :value x}]}))

;(def tests (select-keys tests [:arithmetic]))

(defn run-and-check [program test-desc]
  (let [values
        (loop [{:keys [values state coeffects]} (naive/run program)]
          (if (seq coeffects)
            (let [[id definition] (first coeffects)
                  v (or (get (:blocks test-desc) definition)
                        (throw (ex-info "Unexpected coeffect" {:definition definition})))]
              (recur (naive/unblock program state id v)))
            values))]
    (loop [[e & expectations] (:expectations test-desc) values values]
      (cond
        (and (nil? e) (empty? values)) :ok
        (and e (empty? values)) (throw (ex-info "Value expected" {:expectation e}))
        (and (nil? e) (seq values)) (throw (ex-info "Unexpected value" {:values values}))
        :else (case (:type e)
                :basic (if (= (:value e) (first values))
                         (recur expectations (rest values))
                         (throw (ex-info "Unexpected value" {:value       (first values)
                                                             :expectation e})))
                :permutable (if (contains? (:values e) (first values))
                              (if (= 1 (count (:values e)))
                                (recur expectations (rest values))
                                (recur (cons (update e :values disj (first values)) expectations) (rest values)))
                              (throw (ex-info "Unexpected value" {:value       (first values)
                                                                  :expectation e}))))))))

(deftest test-suits
  (doseq [[suite tests] tests]
    (testing (str suite)
      (doseq [[i [t desc]] (map-indexed vector tests)]
        (prn "---T" t)
        (let [parsed   (parser/parse t)
              analyzed (analyzer/analyze parsed lib/prelude)
              program  (naive/compile analyzed)]
          (prn "---P" program)
          (try
            (run-and-check program (normalize-desc desc))
            (catch clojure.lang.ExceptionInfo e
              (prn e) (throw e))))))))