(ns doppelganger.api
  (:refer-clojure :exclude [filter sync])
  (:require
   [datomic.api :as d])
  (:import
   (java.util WeakHashMap)))

;; Cache keys are either a db (used for d/with databases), or a d/basis-t
;; number (used for dbs which have been transacted to the connection).
;; basis-t -> Returns the t of the most recent transaction reachable via this DB VALUE.
;;
;;   basis-t      63  64  65  66  67
;;   transacted   a___b___c___d___e
;;   d/with            \__f___g
;;     ...                 \__h___i
;;
;; Databases a-e were created by calling d/transact (perhaps outside of our
;; process, and we received the data through Datomic's peer mechanism).
;; Databases f-i were created through our \"with\" wrapper, and exist
;; only in memory.
;;
;; Note that c and f, d g and h, e and i all have the same basis-t number,
;; since the same number of transactions have been committed to them, even
;; though they have different data.  This means that our cache key needs to
;; differentiate them, or we'll get the wrong cache value for one of them.
;;
;; In these cases, we use the db value itself as the cache key.  It's
;; possible that Datomic will create two db values which refer to the same
;; logical db value, but it doesn't seem to in practice.  In any case, as
;; long as it usually doesn't, our cache will save us time.
;;
;; When we get back to the \"transacted\" timeline (a-e), we use basis-t
;; numbers as cache keys.  This is necessary because we don't always have the
;; intermediate db values, such as when data is transacted outside the process.

(doseq [[k v] (ns-publics 'datomic.api)
        :when (not= k 'with)]
  (intern *ns* (with-meta k (meta v)) @v))

(def ^:private dbs
  "Associated information for each db

  If the db was produced with d/with, we keep the following:
     ::predecessor  the db that was passed to d/with
     ::tx-data      delta from predecessor db
     ::caches       map of cache-ids to cache values"
  (WeakHashMap.))

(def ^:private connections
  "Associated information for each connection

  Keys are [cache-id basis-t] and values are the computed
  cache values."
  (WeakHashMap.))

(defn- weak-get [m k]
  (locking m
    (.get m k)))

(defn- weak-update! [m k f & args]
  (locking m
    (.put m k (apply f (.get m k) args))))

(defn- cache-get
  [conn cache-id key]
  (if (number? key)
    (get (weak-get connections conn) [cache-id key] ::no-value)
    (get-in (weak-get dbs key) [::caches cache-id] ::no-value)))

(defn- cache-put!
  [conn cache-id key value]
  (if (number? key)
    (weak-update! connections conn assoc [cache-id key] value)
    (weak-update! dbs key assoc-in [::caches cache-id] value)))

(defn- ->cache-key
  "Convert to the key used to cache a value associated with this db.

  If this db has a predecessor, then it is a d/with database and is
  its own cache key.  If it does not, then it has been transacted and
  its cache key is the basis-t.  The parameter might already be a cache
  key."
  [maybe-cache-key]
  (cond-> maybe-cache-key
    (and (not (number? maybe-cache-key))
         (nil? (::predecessor (weak-get dbs maybe-cache-key))))
    d/basis-t))

(defn- db-predecessor-key [db]
  (->cache-key (::predecessor (weak-get dbs db))))

(defn- predecessor-key
  [cache-key]
  (if (number? cache-key)
    (when (pos? cache-key)
      (dec cache-key))
    (db-predecessor-key cache-key)))

(defn- tx-for-key
  "Tx required to reach this key from its immediate predecessor"
  [conn cache-key]
  (cond
    (not (number? cache-key))
    (::tx-data (weak-get dbs cache-key))

    (zero? cache-key)
    (let [uri (str "datomic:mem://" (d/squuid))
          _ (d/create-database uri)
          conn (d/connect uri)]
      (d/datoms (d/db conn) :eavt))

    :else
    (mapcat :data (d/tx-range (d/log conn) cache-key (inc cache-key)))))

(defn with
  [& args]
  (let [{:keys [db-before db-after tx-data] :as result} (apply d/with args)]
    (weak-update! dbs db-after (constantly {::predecessor db-before
                                            ::tx-data tx-data
                                            ::caches {}}))
    result))
(alter-meta! #'with merge (select-keys (meta #'datomic.api/with) [:arglists :doc]))

(defn- do-get
  [cache-id value-builder initial-value conn cache-key]
  (let [[start-value ks]
        (reduce
         (fn [ks k]
           (if (nil? k)
             (reduced [initial-value ks])
             (let [value (cache-get conn cache-id k)]
               (if (not= value ::no-value)
                 (reduced [value ks])
                 (cons k ks)))))
         (list)
         (iterate predecessor-key cache-key))]
    (reduce
     (fn [value k]
       (let [tx (tx-for-key conn k)
             value' (value-builder value tx)]
         (cache-put! conn cache-id k value')
         value'))
     start-value
     ks)))

(defn make-getter
  "Make a function for getting a value associated with a db.

  If the value has already been cached for the db, fine, we
  just return it.  Otherwise, we find the most recent cached
  value we can use and the tx which occurred from that until
  our db, and ask value-builder to compute the new value
  (which we then cache for next time)."
  [value-builder initial-value]
  (let [cache-id (gensym)]
    (fn getter* [conn db]
      (do-get cache-id value-builder initial-value conn (->cache-key db)))))
