(ns ^{:author "Stanislas Nanchen"
      :doc    "Codesmith Logger is a simple wrapper on logback (slf4j) and net.logstash.logback.
               To use the library, it is necessary to call the macro `deflogger` before any logging
               macro is called."}
  codesmith.logger.core
  (:require [cheshire.core]
            [clojure.pprint :as pp]
            [clojure.string :as str]
            [cheshire.core :as json])
  (:import [org.slf4j LoggerFactory Logger Marker]
           [net.logstash.logback.marker RawJsonAppendingMarker]
           [clojure.lang RT]))

;; # Configuration

(def default-context-logging-key "context")

(def context-logging-key
  "The key under which the [[log-c]] and [[log-e]] macros put the context in the logstash JSON.
  By default, it is \"context\"."
  default-context-logging-key)

(defn set-context-logging-key!
  "Configuration function to set the [[context-logging-key]]."
  [logging-key]
  (alter-var-root #'context-logging-key (constantly logging-key)))

(def default-context-pre-logging-transformation identity)

(def context-pre-logging-transformation
  "A transformation function that is applied to the context before logging.
  By default, it is the identity (no transformation)."
  default-context-pre-logging-transformation)

(defn set-context-pre-logging-transformation!
  "Configuration function to set the [[context-pre-logging-transformation]]"
  [tranformation]
  (alter-var-root #'context-pre-logging-transformation (constantly tranformation)))

;; # Logger definition

(defmacro deflogger
  "Creates a var named `⠇⠕⠶⠻` in the current namespace `*ns*` to contain a [[org.slf4j.Logger]]
  The name is \"logger\" written in braille-2 notation."
  []
  `(defonce ~(vary-meta '⠇⠕⠶⠻ assoc :tag Logger) (LoggerFactory/getLogger ~(.toString *ns*))))

(deflogger)

;; # Logging macros

(defmacro coerce-string
  "Coerce, at compile time, the argument to be a String."
  [arg]
  (if (instance? String arg)
    arg
    `(str ~arg)))

(defmacro box [arg]
  "Ensures at compile time that the argument is an Object.
  This macro is necessary to avoid reflection warning from numeric constants that are emitted unboxed by the compiler."
  `(RT/box ~arg))

(defmacro compile-to-array [& args]
  "Compile code to create a java array with the given arguments"
  (let [assignments (map-indexed (fn [i arg]
                                   `(aset ~i (box ~arg)))
                                 args)]
    `(doto (object-array ~(count args))
       ~@assignments)))

(defn ensure-throwable ^Throwable [e]
  "Ensures, at runtime, that the argument is a throwable.
  If the argument is not throwable, its string representation is embedded in an `ex-info`."
  (if (instance? Throwable e)
    e
    (let [e-str (str e)]
      (.warn ⠇⠕⠶⠻ "Value {} is not a throwable; wrapping in ex-info" e-str)
      (ex-info e-str {}))))

(defn ^Marker ctx-marker [ctx]
  "Creates a [[RawJsonAppendingMarker]] to include `ctx` in the JSON map produced by the logstash encoder.
  This function attempts first to transform `ctx` in JSON with cheshire; if it fails, it is transformed
  as JSON string with [[pr-str]]."
  (let [ctx   (context-pre-logging-transformation ctx)
        value (try
                (json/generate-string ctx)
                (catch Exception e
                  (.warn ⠇⠕⠶⠻ "Serialization error" ^Exception e)
                  (json/generate-string (pr-str ctx))))]
    (RawJsonAppendingMarker. context-logging-key value)))

(defn level-pred [method]
  "For the given `method` symbol, compute the corresponding `isXXXEnabled` method symbol."
  (let [method-name (name method)]
    (symbol
      (str
        "is"
        (str/upper-case (subs method-name 0 1))
        (subs method-name 1)
        "Enabled"))))

(defn throw-logger-missing-exception []
  "Throw an [[IllegalStateException]] to signal that the `deflogger` macro has not been called properly."
  (throw (IllegalStateException. (str "(deflogger) has not been called in current namespace `" *ns* "`"))))

(defmacro log-c
  "Logging macro to output the context map `ctx` in the JSON string generated by the logstash encoder.
  Variants allow to pass a message as slf4j format string with n arguments.
  The argument `method` is the symbol of the log method to call on the [[Logger]] object. Typically,
  the level macros (`trace-c`, `debug-c`, etc.) are used instead of this macro.
  The macro generates code that verifies that the corresponding log level is enabled."
  ([method ctx]
   (if (resolve '⠇⠕⠶⠻)
     `(if (. ~'⠇⠕⠶⠻ ~(level-pred method))
        (. ~'⠇⠕⠶⠻
           (~method (ctx-marker ~ctx) "")))
     (throw-logger-missing-exception)))
  ([method ctx msg]
   (if (resolve '⠇⠕⠶⠻)
     `(if (. ~'⠇⠕⠶⠻ ~(level-pred method))
        (. ~'⠇⠕⠶⠻
           (~method
             (ctx-marker ~ctx)
             (coerce-string ~msg))))
     (throw-logger-missing-exception)))
  ([method ctx msg & args]
   (if (resolve '⠇⠕⠶⠻)
     (case (count args)
       0 `(if (. ~'⠇⠕⠶⠻ ~(level-pred method))
            (. ~'⠇⠕⠶⠻
               (~method
                 (ctx-marker ~ctx)
                 (coerce-string ~msg))))
       1 `(if (. ~'⠇⠕⠶⠻ ~(level-pred method))
            (. ~'⠇⠕⠶⠻
               (~method
                 (ctx-marker ~ctx)
                 (coerce-string ~msg)
                 (box ~(first args)))))
       2 `(if (. ~'⠇⠕⠶⠻ ~(level-pred method))
            (. ~'⠇⠕⠶⠻
               (~method
                 (ctx-marker ~ctx)
                 (coerce-string ~msg)
                 (box ~(first args))
                 (box ~(second args)))))
       `(if (. ~'⠇⠕⠶⠻ ~(level-pred method))
          (. ~'⠇⠕⠶⠻
             (~method
               (ctx-marker ~ctx)
               (coerce-string ~msg)
               (compile-to-array ~@args)))))
     (throw-logger-missing-exception))))

(defmacro log-m [method msg & args]
  "Logging macro for a simple logging message `msg`.
  Variants allow to pass a message as slf4j format string with n arguments.
  The argument `method` is the symbol of the log method to call on the [[Logger]] object. Typically,
  the level macros (`trace-m`, `debug-m`, etc.) are used instead of this macro.
  The macro generates code that verifies that the corresponding log level is enabled."
  (if (resolve '⠇⠕⠶⠻)
    (case (count args)
      0 `(. ~'⠇⠕⠶⠻
            (~method
              ~msg))
      1 `(. ~'⠇⠕⠶⠻
            (~method
              (coerce-string ~msg)
              (box ~(first args))))
      2 `(. ~'⠇⠕⠶⠻
            (~method
              (coerce-string ~msg)
              (box ~(first args))
              (box ~(second args))))
      `(. ~'⠇⠕⠶⠻
          (~method
            (coerce-string ~msg)
            (compile-to-array ~@args))))
    (throw-logger-missing-exception)))

(defmacro log-e
  "Logging macro for logging a [[Throwable]] using the dedicated logging methods for errors.
  Variants allow to pass an explicit message `msg` and an explicit context `ctx`. If no `msg` is
  provided, the message of the [[Throwable]] is use instead. If no context is provided, we use
  `ex-data` to obtain a context. If a context is provided, it is merged with `ex-data`.
  The argument `method` is the symbol of the log method to call on the [[Logger]] object. Typically,
  the level macros (`trace-e`, `debug-e`, etc.) are used instead of this macro.
  The macro generates code that verifies that the corresponding log level is enabled."
  ([method e]
   `(let [e#   (ensure-throwable ~e)
          msg# (.getMessage e#)]
      (log-e ~method e# msg#)))
  ([method e msg]
   (if (resolve '⠇⠕⠶⠻)
     `(if (. ~'⠇⠕⠶⠻ ~(level-pred method))
        (let [e#     (ensure-throwable ~e)
              e-ctx# (ex-data e#)]
          (if e-ctx#
            (. ~'⠇⠕⠶⠻
               (~method
                 (ctx-marker e-ctx#)
                 (coerce-string ~msg)
                 e#))
            (. ~'⠇⠕⠶⠻
               (~method
                 (coerce-string ~msg)
                 e#)))))
     (throw-logger-missing-exception)))
  ([method e ctx msg]
   (if (resolve '⠇⠕⠶⠻)
     `(if (. ~'⠇⠕⠶⠻ ~(level-pred method))
        (let [e#     (ensure-throwable ~e)
              e-ctx# (ex-data e#)
              ctx#   ~ctx]
          (if e-ctx#
            (. ~'⠇⠕⠶⠻
               (~method
                 (ctx-marker (into e-ctx# ctx#))
                 (coerce-string ~msg)
                 ^Throwable e#))
            (. ~'⠇⠕⠶⠻
               (~method
                 (ctx-marker ctx#)
                 (coerce-string ~msg)
                 ^Throwable e#)))))
     (throw-logger-missing-exception))))

(defmacro trace-c
  "Uses `loc-c` on trace level."
  ([ctx]
   `(log-c ~'trace ~ctx))
  ([ctx msg]
   `(log-c ~'trace ~ctx ~msg))
  ([ctx msg & args]
   `(log-c ~'trace ~ctx ~msg ~@args)))

(defmacro trace-m [msg & args]
  "Uses `loc-m` on trace level."
  `(log-m ~'trace ~msg ~@args))

(defmacro trace-e
  "Uses `log-e` on trace level."
  ([e]
   `(log-e ~'trace ~e))
  ([e msg]
   `(log-e ~'trace ~e ~msg))
  ([e ctx msg]
   `(log-e ~'trace ~e ~ctx ~msg)))


(defmacro debug-c
  "Uses `log-c` on debug level."
  ([ctx]
   `(log-c ~'debug ~ctx))
  ([ctx msg]
   `(log-c ~'debug ~ctx ~msg))
  ([ctx msg & args]
   `(log-c ~'debug ~ctx ~msg ~@args)))

(defmacro debug-m [msg & args]
  "Uses `log-m` on debug level."
  `(log-m ~'debug ~msg ~@args))

(defmacro debug-e
  "Uses `log-e` on debug level."
  ([e]
   `(log-e ~'debug ~e))
  ([e msg]
   `(log-e ~'debug ~e ~msg))
  ([e ctx msg]
   `(log-e ~'debug ~e ~ctx ~msg)))


(defmacro info-c
  "Uses `log-c` on info level."
  ([ctx]
   `(log-c ~'info ~ctx))
  ([ctx msg]
   `(log-c ~'info ~ctx ~msg))
  ([ctx msg & args]
   `(log-c ~'info ~ctx ~msg ~@args)))

(defmacro info-m [msg & args]
  "Uses `log-m` on info level."
  `(log-m ~'info ~msg ~@args))

(defmacro info-e
  "Uses `log-e` on info level."
  ([e]
   `(log-e ~'info ~e))
  ([e msg]
   `(log-e ~'info ~e ~msg))
  ([e ctx msg]
   `(log-e ~'info ~e ~ctx ~msg)))


(defmacro warn-c
  "Uses `log-c` on warn level."
  ([ctx]
   `(log-c ~'warn ~ctx))
  ([ctx msg]
   `(log-c ~'warn ~ctx ~msg))
  ([ctx msg & args]
   `(log-c ~'warn ~ctx ~msg ~@args)))

(defmacro warn-m [msg & args]
  "Uses `log-m` on warn level."
  `(log-m ~'warn ~msg ~@args))

(defmacro warn-e
  "Uses `log-e` on warn level."
  ([e]
   `(log-e ~'warn ~e))
  ([e msg]
   `(log-e ~'warn ~e ~msg))
  ([e ctx msg]
   `(log-e ~'warn ~e ~ctx ~msg)))


(defmacro error-c
  "Uses `log-c` on error level."
  ([ctx]
   `(log-c ~'error ~ctx))
  ([ctx msg]
   `(log-c ~'error ~ctx ~msg))
  ([ctx msg & args]
   `(log-c ~'error ~ctx ~msg ~@args)))

(defmacro error-m [msg & args]
  "Uses `log-m` on error level."
  `(log-m ~'error ~msg ~@args))

(defmacro error-e
  "Uses `log-e` on error level."
  ([e]
   `(log-e ~'error ~e))
  ([e msg]
   `(log-e ~'error ~e ~msg))
  ([e ctx msg]
   `(log-e ~'error ~e ~ctx ~msg)))

;; # Spy macro

(defmacro spy
  "A spy macro to log inspection of an expression.
  It will log the value of the expression and the expression itself as context via `log-c`
  with \"spy\" as message.
  By default, it uses the debug level; the diadic version allows to specify the level as string,
  keyword or symbol (e.g. `(spy :info (+ 1 2)) for info level."
  ([expr]
   `(spy :debug ~expr))
  ([level expr]
   (let [method (symbol (name level))]
     `(let [val# ~expr]
        (log-c ~method {:expression (str/trim (with-out-str
                                                (pp/with-pprint-dispatch
                                                  pp/code-dispatch
                                                  (pp/pprint '~expr))))
                        :value      val#}
               "spy")
        val#))))
