(ns coast.db
  (:require [lighthouse.core :as db]
            [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.string :as string]
            [coast.utils :as utils]
            [coast.env :as env])
  (:refer-clojure :exclude [update drop])
  (:gen-class))

(def conn (atom nil))

(defn connect [s]
  (db/connect s))

(defn read-file-migrations []
  (edn/read-string (str "[" (->> "migrations.edn"
                                io/resource
                                slurp)
                        "]")))

(defn read-db-migrations []
  (edn/read-string (str "[" (->> (db/q @conn '[:select coast-migrations/migration])
                                 (map :coast-migrations/migration)
                                 (string/join ""))
                        "]")))

(defn migrate
  ([]
   (migrate {}))
  ([m]
   (let [file-migrations (read-file-migrations)
         db-migrations (set (read-db-migrations))
         migrations (filter #(not (contains? db-migrations %)) file-migrations)]
     (doseq [migration migrations]
       (println (db/migrate @conn migration m))))))

(defn rollback
  ([]
   (db/rollback @conn))
  ([m]
   (db/rollback @conn m)))

(defn q
  ([v params]
   (db/q @conn v params))
  ([v]
   (db/q @conn v)))

(defn defq [filename]
  (db/defq @conn filename))

(defn pull [v where-clause]
  (db/pull @conn v where-clause))

(defn transact
  ([query params]
   (db/transact @conn query params))
  ([query]
   (db/transact @conn query)))

(defn insert [val]
  (db/insert @conn val))

(defn update [val]
  (db/update @conn val))

(defn delete [val]
  (db/delete @conn val))

(defn cols [k]
  (db/cols @conn k))

(def create db/create)
(def drop db/drop)

(defn schema []
  (db/schema @conn))

(defn -main [& args]
  (let [[arg1 arg2] args
        _ (reset! conn (connect (env/env :database-url)))]
    (case arg1
      "create" (create (utils/snake arg2))
      "drop" (drop (utils/snake arg2))
      "migrate" (migrate {:dry-run? (= "--dry-run" arg2)})
      "rollback" (rollback)
      "")))
