(ns coast.db.sqlite.migrator
  (:require [clojure.string :as string]
            [coast.utils :as utils]))

; things
; 1. db/col
; 2. db/rel
; 4. db/index

(defn col? [m]
  (contains? m :db/col))

(defn not-null [m]
  (when (false? (or (:db/nil? m)
                    (:db/nil m)))
    "not null"))

(defn col-default [m]
  (when (contains? m :db/default)
    (str "default " (get m :db/default))))

(defn col-name [m]
  (-> m :db/col name utils/snake))

(defn col [m]
  (->> [(col-name m)
        (:db/type m)
        (not-null m)
        (col-default m)]
       (filter some?)
       (string/join " ")))

(defn add-column [m]
  (let [table (-> m :db/col namespace utils/snake)
        unique (if (or (contains? m :db/unique?)
                       (contains? m :db/unique))
                  (str "create unique index idx_" table "_" (col-name m) " on " table " (" (col-name m) ")")
                  nil)
        statements [(str "alter table " table " add column " (col m))
                    unique]]
    (string/join ";\n"
      (filter some? statements))))

(defn add-columns [v]
  (->> (filter col? v)
       (map add-column)))

(defn rel? [m]
  (contains? m :db/rel))

(defn name* [val]
  (when (ident? val)
    (name val)))

(defn rel [m]
  (let [ref (condp = (:db/type m)
              :many (-> m :db/joins name* utils/snake)
              :one (-> m :db/rel name* utils/snake)
              nil)
        on-delete (str "on delete " (or (:db/delete m) "cascade"))]
    (when (string? ref)
      (str ref " integer not null references " ref "(id) " on-delete))))

(defn create-table-if-not-exists [table]
  (str "create table if not exists " table " ("
       " id integer primary key autoincrement,"
       " updated_at timestamp,"
       " created_at timestamp not null default current_timestamp"
       " )"))

(defn tables [v]
  (->> (map :db/col v)
       (map namespace)
       (map utils/snake)))

(defn sql [m]
  (cond
    (col? m) (add-column m)
    ;(rel? m) (add-rel m)
    :else ""))

(defn migration [v]
  (let [tables-sql (->> (tables v)
                        (map create-table-if-not-exists)
                        (string/join ";\n"))]
    (str tables-sql "\n"
      (->> (map sql v)
           (string/join ";\n")))))

(comment
  (migration [{:db/col :user/name :db/type "text" :db/unique? true}]))
