(ns tomlee.services
  (:require [taoensso.timbre :as log]
            clojure.string))

(defn make-service [my-ns my-fn]
  [my-ns (symbol (str my-ns "/" my-fn))])

(defmacro service
  ([my-ns]
   `(make-service (quote ~my-ns) (quote ~'start)))
  ([my-ns my-fn]
   `(make-service (quote ~my-ns) (quote ~my-fn))))

(defmacro services [args]
  `(map (fn [arg#]
          (cond
            (vector? arg#)
             (make-service (first arg#) (second arg#))
            :else
             (make-service arg# (quote ~'start)))) '~args))

(defmacro defservices [name arg]
  `(def ~name (services ~arg)))

(defn start-service [service conf]
  (let [[service start-fn-name] service
        start-fn (resolve start-fn-name)]
    (try
      (log/info "starting service:" service)
      (if start-fn
        (do
          (let [result [service :started (start-fn conf)]]
            (log/info "started service:" service)
            result))
        (do
          (log/error "unable to resolve:" start-fn-name)
          [service :failed #()]))
      (catch Exception e
        (log/error e "error starting service:" service)
        [service :failed #()]))))

(defn stop-service [live-service]
  (let [[service status stop-fn] live-service]
    (if (= status :started)
      (log/info "stopping service:" service)
      (try
        (stop-fn)
        (log/info "stopped service:" service)
        (catch Exception e
          (log/error e "error stopping service:" service))))))

(defn- services-failed [services]
  (log/warn "the following services failed to start properly:" (clojure.string/join ", " services)))

(defn start-services
  ([conf services]
   (start-services conf services services-failed))
  ([conf services on-error]
   (let [live-services (doall (map #(start-service % conf) services))
         failed-services (filter #(= (second %) :failed) live-services)]
     (if (empty? failed-services)
       (log/info "all services ready")
       (on-error (map first failed-services)))
     live-services)))

(defn stop-services [live-services]
  (doseq [live-service live-services]
    (stop-service live-service)))

