(ns tomlee.services
  (:require [taoensso.timbre :as log]
            clojure.string))

(defn start-service [service conf]
  (let [start-fn-name (str service "/start")
        start-fn (resolve (symbol start-fn-name))]
    (try
      (log/info "starting service:" service)
      (if start-fn
        (do
          (let [result [service :started (start-fn conf)]]
            (log/info "started service:" service)
            result))
        (do
          (log/error "unable to resolve:" start-fn-name)
          [service :failed #()]))
      (catch Exception e
        (log/error e "error starting service:" service)
        [service :failed #()]))))

(defn stop-service [live-service]
  (let [[service status stop-fn] live-service]
    (if (= status :started)
      (log/info "stopping service:" service)
      (try
        (stop-fn)
        (log/info "stopped service:" service)
        (catch Exception e
          (log/error e "error stopping service:" service))))))

(defn- services-failed [services]
  (log/warn "the following services failed to start properly:" (clojure.string/join ", " services)))

(defn start-services
  ([conf services]
   (start-services conf services services-failed))
  ([conf services on-error]
   (let [live-services (doall (map #(start-service % conf) services))
         failed-services (filter #(= (second %) :failed) live-services)]
     (if (empty? failed-services)
       (log/info "all services ready")
       (on-error (map first failed-services)))
     live-services)))

(defn stop-services [live-services]
  (doseq [live-service live-services]
    (stop-service live-service)))

