(ns formic.field
  (:require [formic.validation :as fv]
            [struct.core :as st]
            [cljs.pprint :refer [pprint]]
            [reagent.core :as r]))

;; touch all
;; --------------------------------------------------------------

(defn touch-all! [fields]
  (doseq [f fields]
    (cond
      (or (:compound f)
          (:flex f))
      (touch-all! @(:current-value f))
      :else
      (reset! (:touched f) true))))

;; serialization
;; --------------------------------------------------------------

(declare serialize)

(defn serialize-field [f]
  (cond
    (:compound f)
    (assoc ((:serializer f)
            (serialize @(:current-value f)))
           :_compound (:compound f))
    ;; flexible
    (:flex f)
    (mapv (fn [flex-f]
             (serialize-field flex-f))
          @(:current-value f))
    @(:touched f)
    (let [field-type (:type f)
          serializer (:serializer f)]
      (serializer @(:current-value f)))
    :else nil))

(defn serialize
  ([fields] (serialize {} fields))
  ([result fields]
   (if (nil? fields) result
       (reduce (fn [result field]
                 (if-let  [serialized (serialize-field field)]
                   (assoc result (:id field) serialized)
                   result))
               result fields))))

;; error handling
;; --------------------------------------------------------------

(defn validate-field [validation touched current-value]
  (when (and @touched validation)
    (first (st/validate-single
            @current-value
            validation))))

(defn validate-compound [path validation current-value]
  (let [untouched-removed (into {}
                           (for [[k v] @current-value]
                             (when (:touched v)
                               [k v])))]
      (when validation
        (first
         (st/validate untouched-removed validation)))))

(defn validate-all [fields]
  (loop [fields fields acc {}]
    (if (empty? fields) (not-empty acc)
     (let [f (first fields)]
       (if-let [err (not-empty
                     (cond
                       (:flex f)
                       (validate-all @(:current-value f))
                       (:compound f)
                       (or @(:err f)
                           (validate-all @(:current-value f)))
                       :else @(:err f)))]
         (recur (rest fields) (assoc acc (:id f) err))
         (recur (rest fields) acc))))))

;; field prep
;; --------------------------------------------------------------

(declare prepare-field)

(defn prepare-field-basic [{:keys [parsers defaults] :as form-schema} form-state f path]
  (let [default-value
        (or (:default f)
            (get defaults (:type f))
            (when (and (or (= (:type f) :select)
                           (= (:type f) :radios))
                       (:options f))
              (first (first (:options f)))))
        supplied-value (get-in @form-state path)
        parser (or (:parser f)
                   (get parsers (:type f) identity))
        parsed-value (if supplied-value
                       (parser supplied-value)
                       default-value)
        touched (not (nil? parsed-value))]
    (swap! form-state assoc-in path {:value parsed-value
                                     :touched touched})))

(defn prepare-field-compound [form-schema form-state f path]
  (let [compound-type (:compound f)
        compound-fields (get-in form-schema [:compound compound-type :fields])]
    (doseq [f compound-fields]
      (prepare-field form-schema form-state f (conj path (:id f))))))

(defn prepare-field-flexible [form-schema form-state f path]
  (let [flex-values (get-in @form-state path)]

    (doseq [n (range (count flex-values))
            :let [ff (get flex-values n)]]
      (let [field-type (keyword (:compound ff))
            field-id   (keyword (str (name (:id f)) "-" n "-" (name field-type)))
            new-path   (conj path n)]
        (swap! form-state assoc-in (conj new-path :_key) n)
        (swap! form-state assoc-in (conj new-path :title) field-type)
        (prepare-field form-schema
                       form-state
                       (assoc ff
                              :id field-id
                              :title field-type)
                       new-path)))))

(defn prepare-field [form-schema form-state f path]
  (cond
    (:compound f)
    (prepare-field-compound form-schema form-state f path )
    (:flex f)
    (prepare-field-flexible form-schema form-state f path)
    :else
    (prepare-field-basic form-schema form-state f path)))

(defn prepare-state [form-schema values]
  (let [form-state (r/atom values)]
   (doseq [f (:fields form-schema)]
     (prepare-field form-schema form-state f [(:id f)]))
   form-state))

;; flex
;; --------------------------------------------------------------

(defn add-field [form-schema next f current-value field-type]
  (let [new-field-id (str (name (:id f)) "-" @next "-" (name field-type))
        new-field {:id new-field-id
                   :_key @next
                   :title (name field-type)
                   :compound field-type}]
    (swap! current-value (fnil conj []) new-field)
    (swap! next inc)))
