(ns formic.frontend
  (:require [formic.components.inputs :as formic-inputs]
            [formic.util :as formic-util]
            [formic.field :as formic-field]
            [reagent.core :as r]
            [cljs.pprint :refer [pprint]]
            [cljsjs.react-flip-move]
            [clojure.string :as s]
           ))

(def flip-move (r/adapt-react-class js/FlipMove))

(declare field)

(defn compound-field [form-schema f]
  (let [compound-error @(:err f)]
    (println )
    [:fieldset.formic-compound
     {:class (str
              (name (:compound f))
              (when compound-error " compound-error"))}
     [:h4.formic-compound-title (formic-util/format-kw (or (:title f)
                                                 (:id f)))]
     [:ul.formic-compound-fields
      (doall
       (for [n (range (count @(:current-value f)))]
         ^{:key [(:id f) n]}
         [:li
          [field form-schema (get @(:current-value f) n)]]))]
     (when compound-error
       [:ul.compound-errors
        (for [[id err] compound-error]
          ^{:key id}
          [:li [:h4.error [:strong (formic-util/format-kw id)] ": " err]])])]))

(defn flexible-controls [f n]
  (let [is-first (= n 0)
        is-last (= (-> f :current-value deref count dec) n)]
    [:ul.formic-flex-controls
     [:li.up
      [:a
       {:href "#"
        :class (when is-first "disabled")
        :on-click
        (fn [ev]
          (.preventDefault ev)
          (when-not is-first
            (swap! (:current-value f) formic-util/vswap n (dec n))))}
       "↑"]]
     [:li.down
      [:a
       {:href "#"
        :class (when is-last "disabled")
        :on-click
        (fn [ev]
          (.preventDefault ev)
          (when-not is-last
            (swap! (:current-value f) formic-util/vswap n (inc n))))}
       "↓"]]
     [:li.delete
      [:a {:href "#"
           :on-click
           (fn [ev]
             (.preventDefault ev)
             (swap! (:current-value f) formic-util/vremove n))}
       "✗"]]]))

(defn flexible-field [form-schema f]
  [:fieldset.formic-flex
   [:h4.formic-flex-title (formic-util/format-kw (:id f))]
   [flip-move {:enter-animation "fade"
               :leave-animation "fade"}
    (doall
     (for [n (range (count @(:current-value f)))
           :let [ff (get @(:current-value f) n)]]
       ^{:key (:id ff)}
       [:div.formic-flex-field
        [flexible-controls f n]
        [field form-schema ff]]))]
   [:ul.formic-flex-add
    (for [field-type (:flex f)]
      ^{:key field-type}
      [:li
       [:a.button
        {:href "#"
         :on-click (fn [ev]
                     (.preventDefault ev)
                     (formic-field/add-field form-schema f field-type))}
        [:span.plus "+"] (formic-util/format-kw field-type)]])]])

(defn formic-buttons
  "Renders the buttons for a set of formic fields.
  Each button has
  - :id - id for the button
  - :label - label for the button (optional - defaults to formatted id)
  - :on-click - Action to perform on click.
              - calls preventDefault on event
              - fn receives current form-state _atom_
  "
  [buttons form-state]
  [:div.formic-buttons
   [:ul
    (for [b buttons]
      (when b
       (let [id (formic-util/format-kw (:id b))]
         ^{:key b}
         [:li
          [:button.formic-buttons-button
           {:name     id
            :id       id
            :on-click (fn [ev]
                        (.preventDefault ev)
                        ((:on-click b) form-state))}
           (or (:label b) (s/capitalize id))]])))]])

(defn unknown-field [f]
  [:h4 "Unknown:"
   [:pre (with-out-str
           (pprint f))]])

(defn basic-field [form-schema f]
  (let [form-component
        (or (get-in form-schema [:components (:type f)])
            (get formic-inputs/default-components (:type f))
            unknown-field)]
    [form-component f]))

(defn field [form-schema f]
  [:div.formic-field
   (cond
     (:flex f)
     [flexible-field form-schema f]
     (:compound f)
     [compound-field form-schema f]
     :else
     [basic-field form-schema f])])

(defn formic-fields [form-schema form-state]
  [:div.formic-fields
   (for [n (range (count (:fields form-schema)))]
     ^{:key n}
     [field form-schema (get form-state n)])])
