(ns co.multiply.scoped
  #?(:cljs (:require-macros co.multiply.scoped))
  #?(:clj (:require [co.multiply.scoped.impl :as impl]))
  #?(:clj (:import [java.lang ScopedValue ScopedValue$Carrier])))


#?(:clj (defmacro current-scope
          "Returns the current scope map, or an empty map if no scope is active.

           The scope map contains var->value bindings set via `scoping`.

           CLJ only. Not available in CLJS."
          []
          `(impl/current-scope)))


#?(:clj (defmacro with-scope
          "Execute body with a pre-built scope map. Returns the value of body.

           Unlike `scoping`, this takes a scope map (as returned by `current-scope`)
           rather than a bindings vector. Useful for restoring a previously captured
           scope in a different execution context.

           CLJ only. Not available in CLJS."
          [scope & body]
          `(-> (ScopedValue/where impl/scope ~scope)
             (ScopedValue$Carrier/.call
               (fn scope-call# [] ~@body)))))


#?(:clj (defmacro scoping
          "Execute body with additional scoped bindings. Returns the value of body.

           Bindings are merged into the current scope. Use `current-scope` to capture
           the active scope and `with-scope` to restore it in another context (e.g.,
           a virtual thread or callback). In CLJS, falls back to `binding`.

           Example:
             (scoping [*user-id* 123
                       *request-id* \"abc\"]
               (ask *user-id*))  ; => 123

           Scopes can be nested; inner bindings shadow outer ones for the same var."
          [bindings & body]
          (assert (vector? bindings) "`bindings` must be a vector.")
          (assert (even? (count bindings)) "`bindings` must contain an even number of forms.")
          (if (:ns &env)
            `(binding ~bindings ~@body)
            `(-> (ScopedValue/where impl/scope (impl/merge-scope ~(impl/resolve-bindings bindings)))
               (ScopedValue$Carrier/.call
                 (fn scope-call# [] ~@body))))))


#?(:clj (defmacro ask
          "Access a scoped value by symbol.

           Returns the value bound via `scoping` if present, otherwise falls back
           to the var's root binding. Throws if the var is unbound and not in scope.
           In CLJS, returns the var's current binding value.

           Example:
             (def ^:dynamic *user-id* :default)

             (scoping [*user-id* 123]
               (ask *user-id*))  ; => 123

             (ask *user-id*)     ; => :default (falls back to var value)"
          [sym]
          (if (:ns &env)
            sym
            `(impl/get-scoped-var ~(resolve sym)))))
