(ns co.multiply.scoped.impl
  (:import
    [clojure.lang Var$Unbound]
    [java.lang ScopedValue ScopedValue$Carrier]))


(defonce ^{:doc "The ScopedValue instance holding the current scope map.

   Stores a map of var->value bindings established via `scoping`. ScopedValues
   provide efficient, inheritable context for virtual threads without the overhead
   of Clojure's thread-local binding mechanism."}
  scope
  (ScopedValue/newInstance))


(defn current-scope
  "Returns the current scope map, or an empty map if no scope is active.

   The scope map contains var->value bindings set via `scoping`. This is
   primarily for internal use; prefer `ask` for accessing individual values."
  []
  (ScopedValue/.orElse scope {}))


(defn get-scoped-var
  "Retrieve a scoped value, falling back to the var's root binding.

   If the var is in the current scope, returns the scoped value.
   If not in scope, returns the var's current value.
   If the var is unbound and not in scope, throws IllegalStateException.

   This is the runtime implementation for the `ask` macro."
  [v]
  (let [scope (current-scope)]
    (if (contains? scope v)
      ;; A value is available in the scope.
      (get scope v)
      ;; No value in the given scope; attempt to use default value.
      (let [value @v]
        (if (instance? Var$Unbound value)
          (throw (IllegalStateException. (str "Unbound: " v)))
          value)))))


(defn- convert-var
  "Convert even-indexed elements (keys) to var references at compile time.
   Used by `scoping` to transform [sym val sym val] into [(var sym) val ...]."
  [idx sym]
  (if (even? idx)
    (or (resolve sym) (throw (IllegalArgumentException. (str "Cannot resolve: " sym))))
    sym))


(defn resolve-bindings
  "Resolve symbols to vars in a bindings vector at compile time.

   Transforms [sym1 val1 sym2 val2 ...] into [#'sym1 val1 #'sym2 val2 ...].
   Used by `scoping` to convert user-provided symbols to var references."
  [bindings]
  (into [] (map-indexed convert-var) bindings))


(defn merge-scope
  "Merge new bindings into the current scope.

   Takes a flat vector of [var1 val1 var2 val2 ...] and returns a new scope map
   with these bindings added to (or overriding) the current scope. Used internally
   by `scoping` to build the new scope map."
  [bindings]
  (let [binding-count (count bindings)]
    (loop [scope (transient (current-scope))
           k-idx (unchecked-int 0)]
      (if (< k-idx binding-count)
        (let [v-idx (unchecked-inc-int k-idx)]
          (recur
            (assoc! scope (get bindings k-idx) (get bindings v-idx))
            (unchecked-inc-int v-idx)))
        (persistent! scope)))))