(ns co.multiply.pathling
  "Stack-safe path finding and updating for nested data structures.

   Core functions:
   - `path-when` - find all values matching a predicate, with navigation structure
   - `find-when` - find values without navigation (more efficient for read-only)
   - `transform-when` - transform all matching values in place
   - `update-paths` - apply function to locations identified by navigation

   Designed for large data structures (10,000+ matches) where traditional
   recursive approaches would overflow the stack."
  (:require
    [co.multiply.pathling.util :refer [acc-size accumulator accumulator->vec]]
    #?(:cljs [co.multiply.pathling.impl :as impl]))
  #?(:clj (:import [co.multiply.pathling Nav Nav$Updatable Scanner])))


;; ## REMOVE sentinel
;; ################################################################################
;; When a transform function returns REMOVE, the element is removed from
;; its parent collection rather than being replaced.
(def REMOVE
  "Sentinel value that signals removal of an element from its parent collection.
   Return this from an update-paths transform function to remove the element.

   Behavior by collection type:
   - Maps: the key-value pair is removed (dissoc)
   - Sets: the element is not added to the result
   - Vectors/Lists: the element is removed and indices collapse

   Example:
     (transform-when data number? #(if (neg? %) REMOVE (inc %)))
     ;; Removes negative numbers, increments positive ones"
  #?(:clj  Nav/REMOVE
     :cljs impl/REMOVE))

;; # Transformations
;; ################################################################################
(defn update-paths
  "Apply function f to all locations identified in nav within data structure.
   Updates are applied depth-first (children before parents).

   If f returns REMOVE, the element is removed from its parent collection:
   - Maps: key-value pair is dissoc'd
   - Sets: element is not added to result
   - Vectors/Lists: element is removed and indices collapse

   Example:
     (let [{:keys [nav]} (path-when data number?)]
       (update-paths data nav inc))

   For map-based replacement:
     (let [{:keys [matches nav]} (path-when data pred)
           replacements (zipmap matches (map transform matches))]
       (update-paths data nav #(get replacements % %)))"
  [data nav f]
  (if nav
    #?(:clj  (Nav$Updatable/.applyUpdates ^Nav$Updatable nav data f REMOVE)
       :cljs (impl/apply-updates nav data f))
    data))


(defn path-when
  "Find all values in nested data structure matching predicate.

   Returns map with:
   - :matches - Vector of matching values in depth-first order
   - :nav     - Navigation structure for updating matches (use with update-paths)

   Returns nil if no matches found.

   Options:
   - :include-keys - When true, also match map keys (default: false)

   Examples:
     (path-when [1 {:a 2} {:b #{3 {:c 4}}}] number?)
     ;=> {:matches [1 2 3 4], :nav ...}

     (path-when [:a :b :c] number?)
     ;=> nil

   Stack-safe for large data structures (10,000+ matches)."
  ([data pred]
   (path-when data pred nil))
  ([data pred opts]
   (let [matches (accumulator)]
     (when-some [nav #?(:clj  (Scanner/pathWhen data matches pred (boolean (:include-keys opts)))
                        :cljs (impl/path-when data matches pred opts))]
       {:matches (accumulator->vec matches)
        :nav     nav}))))


(defn find-when
  "Find all values in nested data structure matching predicate.

   Returns vector of matching values in depth-first order, or nil if no matches.

   The third parameter can be either:
   - A function to transform each match before collecting (default: identity)
   - An options map with keys:
     :tf - Transform function to apply to matches (default: identity)
     :include-keys - When true, also match and collect map keys (default: false)

   More efficient than path-when since it skips navigation structure construction.

   Examples:
     (find-when [1 {:a 2} {:b #{3 {:c 4}}}] number?)
     ;=> [1 2 3 4]

     (find-when [1 {:a 2} {:b #{3 {:c 4}}}] number? inc)
     ;=> [2 3 4 5]

     (find-when {:foo :bar :baz :qux} keyword? {:include-keys true})
     ;=> [:bar :foo :qux :baz]  ; order varies

   Stack-safe for large data structures (10,000+ matches)."
  ([data pred]
   (find-when data pred identity))
  ([data pred tf-or-opts]
   (let [matches (accumulator)
         opts    (if (map? tf-or-opts)
                   (update tf-or-opts :tf #(or % identity))
                   {:tf tf-or-opts})]
     #?(:clj  (Scanner/findWhen data matches pred (:tf opts) (boolean (:include-keys opts)))
        :cljs (impl/find-when data matches pred opts))
     (when-not (zero? (acc-size matches))
       (accumulator->vec matches)))))


(defn transform-when
  "Transform all instances of items matching `pred`.

   If tf returns REMOVE, the element is removed from its parent collection.
   Returns data unchanged if no matches found.

   Options:
     :include-keys - When true, also transform map keys that match pred (default: false)

   Examples:
     (transform-when data number? inc)
     (transform-when data map? #(assoc % :processed true))
     (transform-when {:a 1 :b 2} keyword? name {:include-keys true})
     ;=> {\"a\" 1, \"b\" 2}
     (transform-when [1 -2 3 -4] number? #(if (neg? %) REMOVE (inc %)))
     ;=> [2 4]"
  ([data pred tf]
   (transform-when data pred tf nil))
  ([data pred tf opts]
   (let [{:keys [nav]} (path-when data pred (update opts :include-keys true?))]
     (update-paths data nav tf))))
