package co.multiply.pathling;

import clojure.lang.*;
import java.util.ArrayList;

/**
 * Navigation structures for Pathling.
 *
 * These types record the path to matching elements in a data structure,
 * enabling efficient targeted updates without re-scanning.
 *
 * All navigation types implement {@link Updatable} for efficient virtual dispatch
 * during the update phase.
 */
public final class Nav {
    private Nav() {} // Prevent instantiation

    /**
     * Sentinel value indicating an element should be removed from its parent collection.
     * Using a unique object ensures identity comparison works correctly.
     */
    public static final Object REMOVE = new Object() {
        @Override
        public String toString() {
            return "Nav.REMOVE";
        }
    };

    /**
     * Interface for applying updates through navigation structure.
     * Implemented by all Nav types for efficient virtual dispatch.
     */
    public sealed interface Updatable
        permits MapEditable, MapPersistent, MapStruct, VecEdit, VecPersistent,
                SeqNav, SetEdit, SetPersistent, Scalar {
        /**
         * Apply function f to all locations identified by this navigation.
         *
         * @param data the data structure to update
         * @param f the transform function
         * @param remove sentinel value indicating element should be removed
         * @return the updated data structure
         */
        Object applyUpdates(Object data, IFn f, Object remove);
    }

    // ========================================================================
    // Map entry navigation types (sealed for exhaustive pattern matching)
    // ========================================================================

    /** Navigation for map entries - distinguishes value-only, key-only, and both */
    public sealed interface KeyNav permits Val, Key, KeyVal {}

    /** Navigate into value only (key unchanged) */
    public record Val(Object key, Updatable child) implements KeyNav {}

    /** Transform key only (value unchanged) */
    public record Key(Object key) implements KeyNav {}

    /** Transform key and navigate into value */
    public record KeyVal(Object key, Updatable child) implements KeyNav {}

    // ========================================================================
    // Collection-level navigation types
    // ========================================================================

    /**
     * Navigation for editable maps (supports transients).
     * Can contain Val, Key, or KeyVal entries.
     */
    public record MapEditable(ArrayList<KeyNav> children, boolean terminal) implements Updatable {
        @Override
        public Object applyUpdates(Object data, IFn f, Object remove) {
            IPersistentMap m = (IPersistentMap) data;
            Object updated;
            if (children != null) {
                ITransientMap tm = (ITransientMap) ((IEditableCollection) m).asTransient();
                int n = children.size();
                // First pass: remove all keys that will be modified
                for (int i = 0; i < n; i++) {
                    Object k = switch (children.get(i)) {
                        case Val v -> v.key();
                        case Key k2 -> k2.key();
                        case KeyVal kv -> kv.key();
                    };
                    tm = tm.without(k);
                }
                // Second pass: add back with potentially new keys and values
                for (int i = 0; i < n; i++) {
                    switch (children.get(i)) {
                        case Val v -> {
                            Object value = v.child().applyUpdates(RT.get(m, v.key()), f, remove);
                            if (value != remove) {
                                tm = tm.assoc(v.key(), value);
                            }
                        }
                        case Key k -> {
                            Object newK = f.invoke(k.key());
                            if (newK != remove) {
                                tm = tm.assoc(newK, RT.get(m, k.key()));
                            }
                        }
                        case KeyVal kv -> {
                            Object newK = f.invoke(kv.key());
                            Object value = kv.child().applyUpdates(RT.get(m, kv.key()), f, remove);
                            if (newK != remove && value != remove) {
                                tm = tm.assoc(newK, value);
                            }
                        }
                    }
                }
                updated = withMeta(tm.persistent(), RT.meta(m));
            } else {
                updated = m;
            }
            return terminal ? f.invoke(updated) : updated;
        }
    }

    /**
     * Navigation for non-editable maps (no transient support).
     * Can contain Val, Key, or KeyVal entries.
     */
    public record MapPersistent(ArrayList<KeyNav> children, boolean terminal) implements Updatable {
        @Override
        public Object applyUpdates(Object data, IFn f, Object remove) {
            IPersistentMap m = (IPersistentMap) data;
            Object updated;
            if (children != null) {
                // First pass: remove all keys that will be modified
                IPersistentMap result = m;
                int n = children.size();
                for (int i = 0; i < n; i++) {
                    Object k = switch (children.get(i)) {
                        case Val v -> v.key();
                        case Key k2 -> k2.key();
                        case KeyVal kv -> kv.key();
                    };
                    result = result.without(k);
                }
                // Second pass: add back with potentially new keys and values
                for (int i = 0; i < n; i++) {
                    switch (children.get(i)) {
                        case Val v -> {
                            Object value = v.child().applyUpdates(RT.get(m, v.key()), f, remove);
                            if (value != remove) {
                                result = result.assoc(v.key(), value);
                            }
                        }
                        case Key k -> {
                            Object newK = f.invoke(k.key());
                            if (newK != remove) {
                                result = result.assoc(newK, RT.get(m, k.key()));
                            }
                        }
                        case KeyVal kv -> {
                            Object newK = f.invoke(kv.key());
                            Object value = kv.child().applyUpdates(RT.get(m, kv.key()), f, remove);
                            if (newK != remove && value != remove) {
                                result = result.assoc(newK, value);
                            }
                        }
                    }
                }
                updated = withMeta(result, RT.meta(m));
            } else {
                updated = m;
            }
            return terminal ? f.invoke(updated) : updated;
        }
    }

    /**
     * Navigation for struct maps (fixed keys, no dissoc, no transients).
     * Only contains Val entries since struct maps have fixed keys.
     */
    public record MapStruct(ArrayList<Val> children, boolean terminal) implements Updatable {
        @Override
        public Object applyUpdates(Object data, IFn f, Object remove) {
            IPersistentMap m = (IPersistentMap) data;
            Object updated;
            if (children != null) {
                IPersistentMap result = m;
                int n = children.size();
                for (int i = 0; i < n; i++) {
                    Val v = children.get(i);
                    Object value = v.child().applyUpdates(RT.get(m, v.key()), f, remove);
                    // Struct maps don't support dissoc, so REMOVE becomes assoc nil
                    result = result.assoc(v.key(), value == remove ? null : value);
                }
                updated = result;
            } else {
                updated = m;
            }
            return terminal ? f.invoke(updated) : updated;
        }
    }

    /**
     * Navigation for editable vectors (supports transients).
     */
    public record VecEdit(ArrayList<Pos> children, boolean terminal) implements Updatable {
        @Override
        public Object applyUpdates(Object data, IFn f, Object remove) {
            IPersistentVector v = (IPersistentVector) data;
            Object updated;
            if (children != null) {
                ITransientVector tv = (ITransientVector) ((IEditableCollection) v).asTransient();
                ArrayList<Integer> removals = null;
                int n = children.size();
                for (int i = 0; i < n; i++) {
                    Pos p = children.get(i);
                    Object result = p.child().applyUpdates(v.nth(p.index()), f, remove);
                    if (result == remove) {
                        if (removals == null) removals = new ArrayList<>();
                        removals.add(p.index());
                    } else {
                        tv = tv.assocN(p.index(), result);
                    }
                }
                updated = VecRemover.removeIndices((IPersistentVector) tv.persistent(), removals, RT.meta(v));
            } else {
                updated = v;
            }
            return terminal ? f.invoke(updated) : updated;
        }
    }

    /**
     * Navigation for non-editable vectors (subvec, etc).
     */
    public record VecPersistent(ArrayList<Pos> children, boolean terminal) implements Updatable {
        @Override
        public Object applyUpdates(Object data, IFn f, Object remove) {
            IPersistentVector v = (IPersistentVector) data;
            Object updated;
            if (children != null) {
                IPersistentVector result = v;
                ArrayList<Integer> removals = null;
                int n = children.size();
                for (int i = 0; i < n; i++) {
                    Pos p = children.get(i);
                    Object value = p.child().applyUpdates(v.nth(p.index()), f, remove);
                    if (value == remove) {
                        if (removals == null) removals = new ArrayList<>();
                        removals.add(p.index());
                    } else {
                        result = result.assocN(p.index(), value);
                    }
                }
                updated = VecRemover.removeIndices(result, removals, RT.meta(v));
            } else {
                updated = v;
            }
            return terminal ? f.invoke(updated) : updated;
        }
    }

    /**
     * Navigation for lists and other sequential collections.
     */
    public record SeqNav(ArrayList<Pos> children, boolean terminal) implements Updatable {
        @Override
        public Object applyUpdates(Object data, IFn f, Object remove) {
            // Convert to vector for indexed updates, convert back to list at end
            ISeq s = RT.seq(data);
            IPersistentVector dataVec = PersistentVector.create(s);

            Object updated;
            if (children != null) {
                ITransientVector tv = (ITransientVector) ((IEditableCollection) dataVec).asTransient();
                ArrayList<Integer> removals = null;
                int n = children.size();
                for (int i = 0; i < n; i++) {
                    Pos p = children.get(i);
                    Object result = p.child().applyUpdates(dataVec.nth(p.index()), f, remove);
                    if (result == remove) {
                        if (removals == null) removals = new ArrayList<>();
                        removals.add(p.index());
                    } else {
                        tv = tv.assocN(p.index(), result);
                    }
                }
                IPersistentVector resultVec = VecRemover.removeIndices(
                    (IPersistentVector) tv.persistent(), removals, null);
                // Convert back to PersistentList by consing in reverse order
                IPersistentList list = PersistentList.EMPTY;
                for (int i = resultVec.count() - 1; i >= 0; i--) {
                    list = (IPersistentList) list.cons(resultVec.nth(i));
                }
                updated = withMeta(list, RT.meta(data));
            } else {
                updated = data;
            }
            return terminal ? f.invoke(updated) : updated;
        }
    }

    /**
     * Navigation for editable sets (supports transients).
     */
    public record SetEdit(ArrayList<Mem> children, boolean terminal) implements Updatable {
        @Override
        public Object applyUpdates(Object data, IFn f, Object remove) {
            IPersistentSet set = (IPersistentSet) data;
            Object updated;
            if (children != null) {
                ITransientSet ts = (ITransientSet) ((IEditableCollection) set).asTransient();
                // First remove all members that will be transformed
                int n = children.size();
                for (int i = 0; i < n; i++) {
                    ts = ts.disjoin(children.get(i).member());
                }
                // Then add back transformed values
                for (int i = 0; i < n; i++) {
                    Mem m = children.get(i);
                    Object result = m.child().applyUpdates(m.member(), f, remove);
                    if (result != remove) {
                        ts = (ITransientSet) ts.conj(result);
                    }
                }
                updated = withMeta(ts.persistent(), RT.meta(set));
            } else {
                updated = set;
            }
            return terminal ? f.invoke(updated) : updated;
        }
    }

    /**
     * Navigation for non-editable sets (sorted-set, etc).
     */
    public record SetPersistent(ArrayList<Mem> children, boolean terminal) implements Updatable {
        @Override
        public Object applyUpdates(Object data, IFn f, Object remove) {
            IPersistentSet set = (IPersistentSet) data;
            Object updated;
            if (children != null) {
                // First remove all members that will be transformed
                IPersistentSet result = set;
                int n = children.size();
                for (int i = 0; i < n; i++) {
                    result = result.disjoin(children.get(i).member());
                }
                // Then add back transformed values
                for (int i = 0; i < n; i++) {
                    Mem m = children.get(i);
                    Object value = m.child().applyUpdates(m.member(), f, remove);
                    if (value != remove) {
                        result = (IPersistentSet) result.cons(value);
                    }
                }
                updated = withMeta(result, RT.meta(set));
            } else {
                updated = set;
            }
            return terminal ? f.invoke(updated) : updated;
        }
    }

    // ========================================================================
    // Element-level navigation types (used within collections)
    // ========================================================================

    /**
     * Navigation for positional elements (vectors, lists).
     */
    public record Pos(int index, Updatable child) {}

    /**
     * Navigation for set members.
     */
    public record Mem(Object member, Updatable child) {}

    // ========================================================================
    // Terminal navigation
    // ========================================================================

    /**
     * Singleton representing a terminal match (scalar that matched predicate).
     */
    public record Scalar() implements Updatable {
        public static final Scalar INSTANCE = new Scalar();

        @Override
        public Object applyUpdates(Object data, IFn f, Object remove) {
            return f.invoke(data);
        }

        @Override
        public String toString() {
            return "Nav.SCALAR";
        }
    }

    /** For backwards compatibility */
    public static final Updatable SCALAR = Scalar.INSTANCE;

    // ========================================================================
    // Helper methods
    // ========================================================================

    @SuppressWarnings("unchecked")
    private static <T> T withMeta(Object obj, IPersistentMap meta) {
        if (meta != null && obj instanceof IObj o) {
            return (T) o.withMeta(meta);
        }
        return (T) obj;
    }

    /**
     * Helper class for removing indices from vectors efficiently.
     */
    static final class VecRemover {
        /**
         * Remove elements at specified indices from a vector.
         * Indices must be sorted ascending and unique.
         */
        static IPersistentVector removeIndices(IPersistentVector v, ArrayList<Integer> indices, IPersistentMap meta) {
            if (indices == null || indices.isEmpty()) {
                return meta != null ? (IPersistentVector) ((IObj) v).withMeta(meta) : v;
            }

            int end = v.count();
            int removeCount = indices.size();

            // Build result by copying segments between removed indices
            ITransientCollection result = PersistentVector.EMPTY.asTransient();
            int prevEnd = 0;

            for (int i = 0; i < removeCount; i++) {
                int removeIdx = indices.get(i);
                // Copy segment from prevEnd to removeIdx
                for (int j = prevEnd; j < removeIdx; j++) {
                    result = result.conj(v.nth(j));
                }
                prevEnd = removeIdx + 1;
            }
            // Copy remaining segment
            for (int j = prevEnd; j < end; j++) {
                result = result.conj(v.nth(j));
            }

            IPersistentVector finalResult = (IPersistentVector) result.persistent();
            return meta != null ? (IPersistentVector) ((IObj) finalResult).withMeta(meta) : finalResult;
        }
    }
}
