(ns tech.v3.datatype.export-symbols
  (:require [clojure.java.io :as io]
            [clojure.tools.logging :as log])
  (:import [java.io Writer]))


(defmacro export-symbols
  [src-ns & symbol-list]
  `(do
     (require '~src-ns)
     ~@(->> symbol-list
            (mapv
             (fn [sym-name]
               `(let [varval# (requiring-resolve (symbol ~(name src-ns)
                                                         ~(name sym-name)))
                      var-meta# (meta varval#)]
                  (when-not varval#
                    (throw (ex-info
                            (format "Failed to find symbol '%s' in namespace '%s'"
                                    ~(name sym-name) ~(name src-ns))
                            {:symbol '~sym-name
                             :src-ns '~src-ns})))
                  (when (:macro var-meta#)
                    (throw
                     (ex-info
                      (format "Cannot export macros as this breaks aot: %s"
                              '~sym-name)
                      {:symbol '~sym-name})))
                  (def ~(symbol (name sym-name)) @varval#)
                  (alter-meta! #'~(symbol (name sym-name))
                               merge
                               (-> (select-keys var-meta#
                                                [:file :line :column
                                                 :doc
                                                 :column :tag
                                                 :arglists])
                                   (assoc :export-info {:src-ns '~src-ns
                                                        :src-sym '~sym-name})))))))))

(defn- write!
  ^Writer [^Writer writer data & datas]
  (.append writer (str data))
  (doseq [data datas]
    (when data
      (.append writer (str data))))
  writer)


(defn- writeln!
  ^Writer [^Writer writer strdata & strdatas]
  (.append writer (str strdata))
  (doseq [data strdatas]
    (when data
      (.append writer (str data))))
  (.append writer "\n")
  writer)


(defn- indent!
  ^Writer [^Writer writer ^long n-spaces]
  (dotimes [_sp n-spaces]
    (.append writer \space))
  writer)


(defn- escape-doc-quote
  [^String docstr]
  (when docstr
    (.replace docstr "\"" "\\\"")))


(defn write-api!
  "Used to write an easily-discoverable public api.  This simply looks at ns-publics and
  creates stubs that call through to their defined functions and macros.

  Example:

```clojure
  (write-api! 'tech.v3.datatype-api
              'tech.v3.datatype
              \"src/tech/v3/datatype.clj\"
              ['cast 'reverse])
```"
  [src-ns-symbol dst-ns-symbol dst-clj-file exclusions]
  (log/infof "Generating %s" dst-clj-file)
  (require src-ns-symbol)
  (let [ns-doc (:doc (meta (the-ns src-ns-symbol)))
        pubs (ns-publics src-ns-symbol)
        metas (->> (map meta (vals pubs))
                   (sort-by (comp name :name)))
        require-ns (concat [src-ns-symbol]
                           (->> metas
                                (map (comp :src-ns :export-info))
                                (filter identity)
                                (distinct)
                                (sort-by name)))
        last-req (last require-ns)]
    (with-open [writer (io/writer dst-clj-file)]
      ;;ns declaration
      (-> writer
          (writeln! "(ns " (name dst-ns-symbol))
          (indent! 2)
          (writeln! ";;Autogenerated from " (name src-ns-symbol) "-- DO NOT EDIT")
          (indent! 2)
          (write! "\"")
          (write! (escape-doc-quote ns-doc))
          (writeln! "\"")
          (indent! 2)
          (write! "(:require "))
      (doseq [req require-ns]
        (when-not (= req src-ns-symbol)
          (indent! writer 12))
        (if (= last-req req)
          (write! writer "[" req "])")
          (writeln! writer "[" req "]")))
      (when (seq exclusions)
        (write! writer "\n")
        (indent! writer 2)
        (write! writer "(:refer-clojure :exclude [")
        (apply write! writer (interpose " " (map name exclusions)))
        (write! writer "])"))
      (writeln! writer ")\n")
      (doseq [data metas]
        (if (:arglists data)
          (do
            (let [macro? (:macro data)]
              (if macro?
                (writeln! writer "(defmacro " (:name data))
                (writeln! writer "(defn " (:name data)))
              (when-not (empty? (:doc data))
                (indent! writer 2)
                (write! writer "\"")
                (write! writer (escape-doc-quote (:doc data)))
                (writeln! writer "\""))
              (doseq [arglist (:arglists data)]
                ;;Preprocess the arglist to remove map destructuring and turn anything after
                ;;the '&' into a generic 'args' argument.
                (let [varargs (seq (rest (drop-while #(not= '& %) arglist)))
                      varargs (if varargs ['args] nil)
                      arglist-meta (meta arglist)
                      arglist (->> (concat (take-while #(not= '& %) arglist)
                                           varargs)
                                   (map (fn [arglist-arg]
                                          (cond
                                            (symbol? arglist-arg)
                                            arglist-arg
                                            (map? arglist-arg)
                                            (if (arglist-arg :as)
                                              (let [arg-sym (arglist-arg :as)]
                                                (if (= \_ (nth (name arg-sym) 0))
                                                  (symbol (.substring (name arg-sym) 1))
                                                  arg-sym))
                                              (throw (Exception. (format "%s/%s - destructured maps must have an :as member" src-ns-symbol (:name data)))))))))]
                  (indent! writer 2)
                  (write! writer "(")
                  (when-let [meta arglist-meta]
                    (write! writer "^" (str meta) " "))
                  (writeln! writer (if varargs
                                     (str (vec (concat (butlast arglist)
                                                       ['& (last arglist)])))
                                     (str (vec arglist))))
                  (indent! writer 2)
                  (when macro?
                    (write! writer "`"))
                  (let [[src-ns-symbol fn-name]
                        (if-let [export-info (:export-info data)]
                          [(:src-ns export-info) (:src-sym export-info)]
                          [src-ns-symbol (:name data)])]
                    (if (and varargs (not macro?))
                      (write! writer "(apply " src-ns-symbol "/" fn-name " ")
                      (write! writer "(" src-ns-symbol "/" fn-name " ")))
                  (when (seq arglist)
                    (->> (if macro?
                           (if varargs
                             (concat
                              (map #(symbol (str "~" (name %)))
                                   (butlast arglist))
                              [(symbol (str "~@" (name (last arglist))))])
                             (map #(symbol (str "~" (name %)))
                                  arglist))
                           arglist)
                         (interpose " ")
                         (apply write! writer))))
                (if (= arglist (last (:arglists data)))
                  (write! writer "))")
                  (writeln! writer "))"))))
            (writeln! writer ")\n\n"))
          ;;Else this is a def, not defn or defmacro
          (do
            (write! writer "(def ")
            (when-let [dtag (:tag data)]
              (write! writer "^{:tag '" dtag "} "))
            (write! writer (:name data) " ")
            (if-let [export-info (:export-info data)]
              (write! writer (:src-ns export-info) "/" (:src-sym export-info))
              (write! writer src-ns-symbol "/" (:name data)))
            (writeln! writer ")")))))))


(comment
  (write-api! 'tech.v3.datatype-api
              'tech.v3.datatype
              "src/tech/v3/datatype.clj"
              ['cast 'reverse])
  )
