(ns titanium.intro
  (:require [clojurewerkz.titanium.graph    :as tg]
            [clojurewerkz.titanium.edges    :as te]
            [clojurewerkz.titanium.vertices :as tv]
            [clojurewerkz.titanium.types    :as tt]
            [clojurewerkz.titanium.query    :as tq]
            [ogre.core                      :as oq]))

;; Vertices.md

(tg/open (System/getProperty "java.io.tmpdir"))            

(tg/transact! (tv/create!))

(tg/transact!
 (tv/create! {:name "Michael" :location "Europe"})
 (tv/create! {:name "Zack"    :location "America"}))

(tg/transact!
 (let [p1 (tv/create! {:title "ClojureWerkz" :url "http://clojurewerkz.org"})
       p2 (tv/create! {:title "Titanium"     :url "http://titanium.clojurewerkz.org"})]
   (te/connect! p1 :meaningless p2)
   (te/connect! p1 :meaningful  p2 {:verified-on "February 11th, 2013"})))


(tg/transact! 
 (tt/create-property-key :age Long {:indexed-vertex? true :unique-direction :out}))

(tg/transact! 
 (tv/create! {:name "Zack"   :age 22}))
(tg/transact! 
 (tv/create! {:name "Trent"  :age 22}))
(tg/transact! 
 (tv/create! {:name "Vivian" :age 19}))


(tg/transact!
 (tv/find-by-kv :age 22))

(defn degree-of 
  "Finds the degree of the given vertex."
  [v]
  (tq/count-edges v))

(tg/transact!
 (let [v1 (tv/create!)
       v2 (tv/create!)
       v3 (tv/create!)
       v4 (tv/create!)]
   (te/connect! v1 :link v2)
   (te/connect! v1 :link v3)
   (println (map degree-of [v1 v2 v3 v4]))))

(defn clear-graph! 
 "Clears the graph of all objects."
 []
 (doseq [e (te/get-all-edges)]
   (te/delete! e))
 (doseq [v (tv/get-all-vertices)]
   (tv/delete! v)))

(tg/transact!
 (clear-graph!)    
 (let [v1 (tv/create!)
       v2 (tv/create!)
       v3 (tv/create!)
       v4 (tv/create!)]
   (te/connect! v1 :link v2)
   (te/connect! v1 :link v3)
   (println (count (tv/get-all-vertices)) (count (te/get-all-edges)))
   (clear-graph!)    
   (println (count (tv/get-all-vertices)) (count (te/get-all-edges)))))

(defn average [vs]
  (float (/ (reduce + vs) (count vs))))

(defn average-degree 
  "Finds the average degree of a vertex."
  []
  (average (map degree-of (tv/get-all-vertices))))

(defn complete-graph 
  "Clears the graph and generates a complete graph."
  [n]
  (clear-graph!)
  (let [vs (map #(tv/create! {:i %}) (range n))]      
    (doseq [v vs w vs :when (not= v w)]
      (te/connect! v :link w))))

(tg/open {"storage.backend" "inmemory"})
(doseq [i (range 1 10)]
  (tg/transact!
   (complete-graph i)
   (println i (average-degree))))

(defn random-graph 
  "Clears the graph and generates a random graph."
  [n]
  (clear-graph!)
  (let [vs (map #(tv/create! {:i %}) (range n))]      
    (doseq [v vs w vs :when (and (not= v w) (> 0.5 (rand)))]
      (te/connect! v :link w))))

(defn run-experiment [i]
  (tg/transact!
   (random-graph i)
   (average-degree)))

(doseq [i (range 1 10)]
  (let [results (map run-experiment (take 10 (cycle [i])))]
    (println i (average results))))

;; Edges.md            


(def Zack (tg/transact! (tv/create! {:name "Zack"})))
(def Brooke (tg/transact! (tv/create! {:name "Brooke"})))
(def familial-tie (tg/transact! (te/connect! (tv/refresh Zack)
                                             :brother-to 
                                             (tv/refresh Brooke)))

(tg/open (System/getProperty "java.io.tmpdir"))            


(tg/transact!
 (let [saturn   (tv/create! {:name "Saturn"   :type "titan"})
       jupiter  (tv/create! {:name "Jupiter"  :type "god"})
       hercules (tv/create! {:name "Hercules" :type "demigod"})
       alcmene  (tv/create! {:name "Alcmene"  :type "human"})
       neptune  (tv/create! {:name "Neptune"  :type "god"})
       pluto    (tv/create! {:name "Pluto"    :type "god"})
       sea      (tv/create! {:name "Sea"      :type "location"})
       sky      (tv/create! {:name "Sky"      :type "location"})
       tartarus (tv/create! {:name "Tartarus" :type "location"})
       nemean   (tv/create! {:name "Nemean"   :type "monster"})
       hydra    (tv/create! {:name "Hydra"    :type "monster"})
       cerberus (tv/create! {:name "Cerberus" :type "monster"})]
   (te/connect! neptune :lives sea)
   (te/connect! jupiter :lives sky)
   (te/connect! pluto :lives tartarus)
   (te/connect! jupiter :father saturn)
   (te/connect! hercules :father jupiter)
   (te/connect! hercules :mother alcmene)
   (te/connect! jupiter :brother pluto)
   (te/connect! pluto :brother jupiter)
   (te/connect! neptune :brother pluto)
   (te/connect! pluto :brother neptune)
   (te/connect! jupiter :brother neptune)
   (te/connect! neptune :brother jupiter)
   (te/connect! cerberus :lives tartarus)
   (te/connect! pluto :pet cerberus)
   (te/connect! hercules :battled nemean   {:times 1})
   (te/connect! hercules :battled hydra    {:times 2})
   (te/connect! hercules :battled cerberus {:times 12})))

(tg/transact! 
 (oq/query (tv/find-by-kv :name "Saturn")
           (oq/<-- :father)
           (oq/<-- :father)
           (oq/property :name)
           oq/first-of!))

(tg/transact!
 (oq/query (tv/find-by-kv :name "Hercules")
           (oq/--> :father :mother)
           (oq/property :name)
           oq/into-set!))

(tg/transact! 
 (oq/query (tv/find-by-kv :name "Hercules")
           (oq/--E> :battled)
           (oq/has :times > 1)
           (oq/in-vertex)
           (oq/property :name)
           oq/into-set!))

(tg/transact!
 (oq/query (tv/find-by-kv :name "Hercules")
           (oq/--E> :battled)
           oq/count!))

(tg/transact!
 (let [pluto (first (tv/find-by-kv :name "Pluto"))]
   (oq/query pluto
    (oq/--> :lives)
    (oq/<-- :lives)
    (oq/except [pluto])
    (oq/property :name)
    oq/into-set!)))

(tg/transact! 
 (oq/query (tv/find-by-kv :name "Pluto")
           (oq/--> :brother)
           (oq/as  "god")
           (oq/out :lives)
           (oq/as  "place")
           (oq/select (oq/prop :name))
           oq/all-into-vecs!))

;; types

(tg/transact! 
 (tt/create-property-key :age Integer))

(tg/transact!       
 (tt/create-property-key :name String
                         {:indexed-vertex? true}))

(tg/transact!
 (tt/create-edge-label :first-label))

(tg/transact!
 (tt/create-edge-label :heard-of {:direction "unidirected"                                
                                  :unique-direction :out}))