(ns clojurewerkz.performeter.core
  (:import [java.lang.Character]
           [com.codahale.metrics.jvm GarbageCollectorMetricSet MemoryUsageGaugeSet ThreadStatesGaugeSet FileDescriptorRatioGauge]
           [org.hyperic.sigar Sigar SigarLoader])
  (:require [clojurewerkz.performeter.conversion :refer :all]))

(def instance (Sigar.))

(defn cpu-list
  "List of cpus and their info"
  []
  (map to-map (.getCpuInfoList instance)))

(defn cpu-perc-breakdown
  "Current cpu stats"
  []
  (mapv to-map (.getCpuPercList instance)))

(defn proc-mem
  "Process memory metrics"
  [pid]
  (to-map (.getProcMem instance pid)))

(defn proc-cpu
  "Process memory metrics"
  [pid]
  (assoc (to-map (.getProcCpu instance pid))
    :pid pid))

(defn cpu-perc-combined
  "Current cpu stats"
  []
  (to-map (.getCpuPerc instance)))

(defn mem
  "Current mem stats"
  []
  (to-map (.getMem instance)))

(defn netstat
  "Current netstat stats"
  []
  (to-map (.getNetStat instance)))

(defn netstat
  "Current cpu stats"
  []
  (to-map (.getNetStat instance)))

(defn tcp
  "Tcp stats"
  []
  (to-map (.getTcp instance)))

(defn swap
  "Tcp stats"
  []
  (to-map (.getSwap instance)))

(defn disk-usage
  "Disk usage for certain disk"
  [^String disk]
  (to-map (.getDiskUsage instance disk)))

(defn fs-usage
  "Disk usage for certain disk"
  [^String fs]
  (to-map (.getFileSystemUsage instance fs)))

(defn proc-stat
  "Disk usage for certain disk"
  [^String fs]
  (to-map (.getProcStat instance fs)))


(defn- get-metrics
  ([metric-set remap-list]
     (into {}
           (mapv (fn [[a b]] [(get remap-list a) (.getValue b)]) (.getMetrics metric-set))))
  ([metric-set]
     (into {}
           (mapv (fn [[a b]] [a (.getValue b)]) (.getMetrics metric-set)))))

(def garbage-collector-remap-list
  {"PS-Scavenge.count"  :app.jvm.gc.ps_scavenge_count
   "PS-Scavenge.time"   :app.jvm.gc.ps_scavenge_time
   "PS-MarkSweep.count" :app.jvm.gc.ps_mark_sweep_count
   "PS-MarkSweep.time"  :app.jvm.gc.ps_mark_sweep_time})

(def memory-usage-remap-list
  {"pools.PS-Eden-Space.usage"          :app.jvm.memory.pools_eden_space_usage
   "pools.PS-Old-Gen.usage"             :app.jvm.memory.old_gan_usage
   "pools.Code-Cache.usage"             :app.jvm.memory.code_cache_usage
   "pools.Compressed-Class-Space.usage" :app.jvm.memory.compressed_class_space_usage
   "pools.PS-Survivor-Space.usage"      :app.jvm.memory.pools_survivor_space_usage
   "pools.Metaspace.usage"              :app.jvm.memory.pools_metaspace_usage

   "total.init"                         :app.jvm.memory.total_init
   "total.max"                          :app.jvm.memory.total_max
   "total.used"                         :app.jvm.memory.total_used
   "total.committed"                    :app.jvm.memory.total_commited


   "heap.init"                          :app.jvm.memory.heap_init
   "heap.max"                           :app.jvm.memory.heap_max
   "heap.used"                          :app.jvm.memory.heap_used
   "heap.committed"                     :app.jvm.memory.heap_committed
   "heap.usage"                         :app.jvm.memory.heap_usage

   "non-heap.init"                      :app.jvm.memory.non_heap_init
   "non-heap.max"                       :app.jvm.memory.non_heap_max
   "non-heap.used"                      :app.jvm.memory.non_heap_used
   "non-heap.committed"                 :app.jvm.memory.non_heap_commited
   "non-heap.usage"                     :app.jvm.memory.non_heap_usage})

(defn jvm-gc
  []
  (get-metrics (GarbageCollectorMetricSet.) garbage-collector-remap-list))

(defn jvm-memory-usage
  []
  (get-metrics (MemoryUsageGaugeSet.) memory-usage-remap-list))

(def thread-state-remap-list
  {"new.count"           :app.jvm.thread.new_count
   "terminated.count"    :app.jvm.thread.terminated_count
   "deadlocks"           :app.jvm.thread.deadlocks
   "daemon.count"        :app.jvm.thread.daemon_count
   "timed_waiting.count" :app.jvm.thread.timed_waiting_count
   "blocked.count"       :app.jvm.thread.blocked_count
   "waiting.count"       :app.jvm.thread.waiting_count
   "count"               :app.jvm.thread.count
   "runnable.count"      :app.jvm.thread.runnable_count})

(defn jvm-thread-state
  []
  (get-metrics (ThreadStatesGaugeSet.) thread-state-remap-list))
