(ns clojurewerkz.perfometer.stat
  (:require [clojurewerkz.perfometer.util :refer :all]))

(defmulti parse-stats
  (fn [s]
    (if (.startsWith s "cpu")
      "cpu"
      (first (split-by-spaces s)))))

(defn- find-cpu-num
  [s]
  (try
    (if-let [[_ idx-str] (re-find #"cpu(\d+)" s)]
      (+ 1 (parse-int idx-str))
      0)
    (catch Exception e
      (throw (RuntimeException. (str "Can't parse cpu index from " s))))))

(defmethod parse-stats "cpu"
  [s]
  (let [[c & all] (remove empty? (split-by-spaces s))]
    [[:cpu (find-cpu-num c)]
     (let [[user nice system idle iowait irq softirq] (map parse-int all)]
       {:user    user
        :nice    nice
        :system  system
        :idle    idle
        :iowait  iowait
        :irq     irq
        :softirq softirq})]))

(defn- default-impl
  "Default implementation. Was done this way to avoid String interning, since it blocks
   threads."
  [sym s]
  (let [[c & all] (remove empty? (split-by-spaces s))]
    [[sym] (parse-int (first all))]))

(defmethod parse-stats "intr"
  [s]
  (default-impl :intr s))

(defmethod parse-stats "ctxt"
  [s]
  (default-impl :ctxt s))

(defmethod parse-stats "btime"
  [s]
  (default-impl :btime s))

(defmethod parse-stats "processes"
  [s]
  (default-impl :processes s))

(defmethod parse-stats "procs_running"
  [s]
  (default-impl :procs_running s))

(defmethod parse-stats "procs_blocked"
  [s]
  (default-impl :procs_blocked s))

(defmethod parse-stats :default
  [s]
  nil)

(defn parse-stat
  "Parses linux /proc/stat file.

   Resulting data structure contains hash of:
     :cpu Every CPU is denoted by it's index (0-based).
          Sub-hash contains values for:

          :user    normal processes executing in user mode
          :nice    niced processes executing in user mode
          :system  processes executing in kernel mode
          :idle    idle wait time
          :iowait  waiting for I/O to complete
          :irq     servicing interrupts
          :softirq servicing softirqs

      :intr counts of interrupts serviced since boot time (total)
      :ctxt total number of context switches across all CPUs
      :btime gives the time at which the system booted, in seconds since the Unix epoch.
      :processes number of processes and threads created
      :procs_running number of processes currently running
      :procs_blocked number of processes currently blocked, waiting for I/O to complete"
  [stat-contents]
  (->> (clojure.string/split stat-contents #"\n")
       (map parse-stats )
       (filter identity )
       (reduce (fn [acc [path v]]
                 (assoc-in acc path v)) {})))
