(ns clojurewerkz.gizmo.widget
  "Main namespace for working with Widgets - small, atomic, reusable parts of the page."
  (:require [net.cgrand.enlive-html :as html]
            [clojure.pprint :refer [pprint]]
            [clojure.core.reducers :as r]
            [clojure.string :as str]
            [net.cgrand.xml :as xml]
            [bultitude.core :as bultitude]))

(defn- in?
  "Helper function to check wether certain element is a part of collection"
  [seq elm]
  (get (set seq) elm))

(defn- select-values [map ks]
  (reduce #(conj %1 (map %2)) [] ks))

;;
;; Implementation
;;

(defn default-fetch
  "Default fetch function"
  [a]
  a)

(defn default-view
  "Default view function, renders an empty widget by default"
  [_]
  "")

(defn render*
  "Render an Enlive template"
  [t]
  (apply str (html/emit* t)))

(defn- resolve-widget
  "Resolves widget based on identifier"
  [s]
  (assert (get-in s [:attrs :rel]) "Can't resolve widget name. If it's an top-level widget, please add {:widgets ...} clause to your responder, otherwise add `rel` attribute to widget for proper resolution.")
  (let [rel (get-in s [:attrs :rel])]
    (if-let [widget (try (resolve (symbol rel)) (catch Exception e nil))]
      widget
      (if (fn? rel)
        rel
        (fn [_] rel)))))

(comment

  (if-let [widget (resolve (symbol (get-in s [:attrs :rel])))]
    widget
    (throw (Exception. (str "Can't resolve widget " s ". " (get-in s [:attrs :rel]) " is not found")))))
;;
;; API
;;

(defmacro defwidget
  "Defines a new widget.

   Widget is a reusable entry that represents any part of your website. Examples
   include things like header, login form, user profile, or even a complete page
   within a layout. In some other frameworks, widgets are called partials
   or nested templates.

   Receives two named arguments:
     * :fetch is a function that receives an environment and runs some code, potentially
      involving disk or network I/O. Sometimes `fetch` is used just to get a part of
      environment that's applicable for a particular view.
     * :view is a function that returns a string with HTML elements generated by any
       rendering engine, like Stencil, Hiccup or your own HTML generation library. WE
       recommend using Enlive and snippets to render views."
  [widget-name &{:keys [fetch view] :or {fetch         default-fetch
                                         view          default-view}}]
  (let [opts {:name (keyword widget-name) :fetch fetch :view view}]
    `(def ~(vary-meta widget-name assoc :widget true :opts opts)
       (fn [env#] (~view (~fetch env#))))))

(defmacro layout*
  [source args & forms]
  `(html/snippet* (html/html-resource ~source) ~args ~@forms))

(defmacro deflayout
  "Defines a new layout.

  Layout is a outlining template that's shared between several pages on your
  website. Usually it's a set of common surroundings of an HTML page."
  [layout-name source args & forms]
  `(def ~(vary-meta layout-name assoc :layout true :source source)
     (layout* ~source ~args ~@forms)))

(defn inject-core-widgets
  "Injects core widgets into given layout"
  [html-source widgets]
  (html/flatmap
   (html/transformation
    [:widget] (fn [node]
                (let [^symbol widget-id (get-in node [:attrs :id])]
                  (if-let [rel (get widgets (keyword widget-id))]
                    (assoc-in node [:attrs :rel] rel)
                    node))))
   html-source))

(defmacro transform
  [html-source & body]
  `(html/flatmap
    (html/transformation
     ~@body)
    ~html-source))

(defn maybe-generate-id
  [a]
  (if (nil? a) (gensym) a))

;; Check out if it's possible to cache selectors O_O

;; TODO Add widget cache for widgets that were already rendered in different context so that they wouldn't be re-rendered
(defn interpolate-widgets
  "Interpolates widgets from the source code"
  [html-source env]
  (let [html-source (transform html-source
                               [:widget] (fn [w]
                                           (update-in w [:attrs :id] maybe-generate-id)))
        step-widgets (into {}
                           (filter identity
                                   (pmap (fn [w]
                                           (when-let [widget-fn (resolve-widget w)]
                                             [(get-in w [:attrs :id]) (widget-fn env)]))
                                         (html/select html-source [:widget]))))]
    (html/flatmap
     (html/transformation
      [:widget] (fn [widget]
                  (let [widget-fn (resolve-widget widget)
                        id        (get-in widget [:attrs :id])
                        view      (get step-widgets id)]
                    (if (seq? view)
                      (interpolate-widgets view env)
                      view))))
     html-source)))

(defn all-layouts
  "Get all layouts available in current application"
  []
  (->> (all-ns)
       (map #(vals (ns-interns %)))
       flatten
       (filter #(:layout (meta %)))
       (map #(vector (keyword (:name (meta %))) (var-get %)))
       (into {})))

;;
;; Trace
;;

(def ^:dynamic *trace*)

(defmacro with-trace
  [& body]
  `(binding [*trace* (atom [])]
     ~@body))

(defn pprint-to-str
  [m]
  (let [w (java.io.StringWriter.)]
    (pprint m w)
    (.toString w)))

(defwidget trace-widget
  :view (fn [_]
          (lazy-seq
           (html/html [:div {:class "trace-widget"}
                       [:div {:class "trace-btn" }
                        "Traces >>"]
                       [:div {:class "trace-wrapper" }
                        (if (empty? @*trace*)
                          "No traces available, you can use `clojurewerkz.gizmo.widget/trace` to add traces here."
                          [:ul
                           (for [[loc trace] @*trace*]
                             [:li
                              [:b loc]
                              [:pre [:code (pprint-to-str trace)]]])])]]))))

(defmacro trace
  [x]
  (let [line (:line (meta &form))
        file *file*]
    `(let [x# ~x]
       (swap! *trace* conj [(format "(%s:%s)" ~file ~line) x#])
       x#)))
