(ns clojurewerkz.eventoverse.collector.server
  (:gen-class)
  (:require [orcette.conf          :as oc]
            [clojure.tools.logging :as log]
            [langohr.core          :as lh]
            [monger.core           :as mc]
            [clojurewerkz.quartzite.scheduler        :as quartz.scheduler]
            [clojurewerkz.eventoverse.collector.amqp :as collector.amqp])
  (:use     [clojure.tools.cli])
  (:import [com.mongodb MongoOptions ServerAddress]))


;;
;; Implementation
;;

(def config {})

(defn- exit!
  "Logs an error and exits with non-zero code"
  [s]
  (log/error s)
  (System/exit 1))

(defn initialize-mongodb
  [options]
  (when-let [mongodb-opts (:mongodb options)]
    (let [host                (get-in mongodb-opts [:connection :host])
          ^ServerAddress sa   (mc/server-address host)
          ^MongoOptions  opts (mc/mongo-options :threads-allowed-to-block-for-connection-multiplier 2048 :auto-connect-retry true)
          db                  (:database mongodb-opts)]
      (mc/connect! sa opts)
      (mc/set-db! (mc/get-db db))
      (log/infof "Connected to MongoDB at %s, database name is %s" db host))))

(defn initialize-amqp
  [options]
  (when-let [amqp-opts (:amqp options)]
    (let [vhost     (get-in amqp-opts [:connection :vhost])
          username  (get-in amqp-opts [:connection :username])]
      (log/infof "Connecting to RabbitMQ as %s@%s" username vhost)
      (collector.amqp/connect (:connection amqp-opts))
      (log/info "Connected to RabbitMQ"))))

(defn initialize-quartz
  [options]
  (quartz.scheduler/initialize)
  (quartz.scheduler/start))


(def initializers [initialize-mongodb
                   initialize-amqp
                   initialize-quartz])



;;
;; API
;;

(defn start
  [options]
  (alter-var-root (var config) (constantly options))
  (dorun (map #(% options) initializers)))

(defn -main
  [& args]
  (let [[options positional-args banner] (cli args
                                              ["--environment" "Environment to run in (development, production, et cetera)" :default "development"]
                                              ["--config-file" "Path to configuration file to use"])]
    (if-let [config (oc/config-for options)]
      (do
        (log/infof "Using config file at %s" (oc/config-path-for options))
        (start config))
      (exit! "Please either specify environment using --environment switch or config file path using --config-file switch"))
    nil))
