(ns clojurewerkz.eventoverse.collector.core
  (:require [clojurewerkz.eventoverse.collector.logging :as log]
            [clojurewerkz.eventoverse.collector.store   :as store])
  (:use [validateur.validation :only [validation-set presence-of] :as vld])
  (:import [java.util Date]))

;;
;; Implementation
;;

(def ^{ :const true :doc "Default event attributes, except for the timestamp" }
  default-event-attributes { :tags [] })

(defn merge-defaults
  "Merges event attributes with default attributes"
  [event]
  (merge default-event-attributes { :received_at (Date.) } event))

(declare event-validator)
(defn- process-without-validation
  [{:keys [application environment] :as m}]
  (store/insert-event application environment (dissoc m :application :environment)))



;;
;; API
;;

(def event-validator
  (validation-set
   (presence-of :type)
   (presence-of :application)
   (presence-of :environment)   
   (presence-of :emitted_at)
   (presence-of :tags)))

(defn validate
  "Returns validation errors for given event"
  [m]
  (event-validator m))

(defn valid?
  [m]
  (vld/valid? event-validator m))

(defn process
  [event]
  (let [m (merge-defaults event)]
    (if (valid? m)
      (process-without-validation m)
      (log/invalid-event (:application m) (:environment m) (event-validator m)))))
