(ns clojurewerkz.eventoverse.collector.web.common
  (:use [clojure.java.io :only [resource]])
  (:require [stencil.core :as st]
            [noir.core :refer [url-for]]
            [clojure.string :as cs]
            [clj-time.format :as ctf]
            [clojurewerkz.support.string :refer [maybe-append maybe-prepend]]
            [clojurewerkz.eventoverse.collector.core :refer [custom-attributes]]))



(def time-formatter (:date-hour-minute-second ctf/formatters))

(defn format-tag
  [^String tag ^String app ^String env]
  {:tag tag :path (url-for "/apps/:app/env/:env/events/tagged/:tagline" {:app app :env env :tagline tag})})

(defn format-tags
  [m app env]
  (if-let [tags (seq (:tags m))]
    (assoc m :tags (mapv #(format-tag % app env) tags))
    m))

(defn tags-from
  [^String s]
  (cs/split s #","))

(defn format-emitted-at
  [m]
  (if-let [et (:emitted_at m)]
    (assoc m :emitted_at (ctf/unparse time-formatter (ctf/parse et)))
    m))

(defn format-id
  [m]
  (if-let [id (:_id m)]
    (assoc m :_id (.toString id))
    m))

(defn inject-custom-attributes
  [m]
  (assoc m :custom-attributes (custom-attributes m)))

(defn format-hostname
  [m ^String app ^String env]
  (if-let [h (:hostname m)]
    (assoc m :hostname {:path (url-for "/apps/:app/env/:env/events/on-host/:hostname" {:app app :env env :hostname h}) :hostname h})
    (assoc m :hostname {})))



(def ^{:const true}
  default-layout "templates/layout/application")

(defn in-layout
  ([^String body]
     (in-layout default-layout body {}))
  ([^String body options]
     (in-layout default-layout body options))
  ([^String layout ^String body options]
     (st/render-file layout (merge  options {:content body}))))


(defn script-tag-for
  [^String s]
  (format "<script src='%s' type='text/javascript'></script>" (-> s
                                                                  (maybe-append ".js")
                                                                  (maybe-prepend "/js/"))))

(defn include-js
  [& xs]
  (cs/join (mapv script-tag-for xs)))

(defn- render-custom-row
  "Renders custom row. Finds item-specific template first, if it does not exist, falls back to common overriden partial, if it does not
   exist, renders the default one"
  [item common-partial-path]
  (if (resource (str "custom-templates/" common-partial-path "." (:type item) ".mustache"))
    (st/render-file (str "custom-templates/" common-partial-path "." (:type item)) item)
    (if (resource (str "custom-templates/" common-partial-path ".mustache"))
      (st/render-file (str "custom-templates/" common-partial-path) item)
      (st/render-file (str "templates/" common-partial-path) item))))


(defn custom-rows
  [collection common-partial-path]
  (map #(render-custom-row % common-partial-path) collection))

(defn format-list
  [l app env]
  (->> l
       (mapv #(format-tags % app env))
       (mapv format-id)
       ;; (mapv format-emitted-at)
       (mapv #(format-hostname % app env))
       (mapv inject-custom-attributes)))
