(ns clojurewerkz.eventoverse.collector.store.mongo-event-store
  (:refer-clojure :exclude [count find sort])
  (:require [monger.collection :as mc]
            [monger.core       :as mg]
            [monger.db         :as mdb]
            [monger.query     :refer [with-collection find limit sort fields]]
            [monger.operators :refer [$in]]
            [monger.conversion :refer [to-object-id]]
            [clojure.tools.logging :as log]
            monger.json
            clojurewerkz.support.json
            [clojurewerkz.support.string :refer [maybe-chopr]]
            [clojurewerkz.eventoverse.collector.store :as store])
  (:use clojurewerkz.eventoverse.collector.store.generic-store)
  (:import [com.mongodb MongoOptions ServerAddress]
           [java.util.Date]))

(declare ^:dynamic *store*)

(defn- ensure-indexes-on
  [^String collection]
  (log/infof "Ensuring %s has indexes..." collection)
  (mc/ensure-index collection {:type 1})
  (mc/ensure-index collection {:emitted_at 1})
  (mc/ensure-index collection {:tags 1})
  (mc/ensure-index collection {:hostname 1}))

(defn connect!
  [options]
  (log/infof "Using Mongodb configuration options: %s..." options)
  (let [host     (:host options)
        port     (:port options)
        database (:database options)
        ^ServerAddress sa   (mg/server-address host port)
        ^MongoOptions  opts (mg/mongo-options :threads-allowed-to-block-for-connection-multiplier 2048 :auto-connect-retry true)]
    (mg/connect! sa opts)
    (mg/use-db! database)
    (log/infof "Connected to MongoDB at %s, database name is %s" database host)))

(deftype MongoEventStore [client limit]
  GenericStore
  (setup! [_ options]
    (doseq [{:keys [name environments]} (:applications options)]
      (doseq [e environments]
        (let [collection (store/events-bucket-name-for name e)]
          (ensure-indexes-on collection)))))
  (purge! [_ app env]
    (mc/remove (store/events-bucket-name-for app env) {}))

  (get-count
    [_ app env]
    (mc/count (store/events-bucket-name-for app env) {}))

  (most-recent-from
    [_ app env]
    (first (with-collection (store/events-bucket-name-for app env)
             (fields [:received_at])
             (sort   {:received_at -1})
             (limit  1)))))


(defn init-mongo-event-store
  ([] (let [client nil]
        (init-mongo-event-store client)))
  ([options]
     (if (bound? (var *store*))
       *store*
       (let [client (connect! options)
             store  (MongoEventStore. client 50)]
         (alter-var-root (var *store*) (constantly store))
         *store*))))