(ns clojurewerkz.eventoverse.collector.store
  (:refer-clojure :exclude [count find sort])
  (:require [monger.collection :as mc]
            [monger.db         :as mdb]
            [monger.query     :refer [with-collection find limit sort fields]]
            [monger.operators :refer [$in]]
            [monger.conversion :refer [to-object-id]]
            monger.json
            clojurewerkz.support.json
            [clojurewerkz.support.string :refer [maybe-chopr]])
  (:import java.util.Date))


;;
;; API
;;

(defn events-bucket-name-for
  [^String app ^String env]
  (str app "_" env "_events"))

(defn errors-bucket-name-for
  [^String app ^String env]
  (str app "_" env "_invalid_events"))

(defn purge!
  [^String app ^String env]
  (mc/remove (events-bucket-name-for app env) {}))

(defn count
  ([^String app ^String env]
     (mc/count (events-bucket-name-for app env) {}))
  ([^String app ^String env conditions]
     (mc/count (events-bucket-name-for app env) conditions)))

(defn insert
  [^String coll m]
  (mc/insert coll m))

(defn insert-event
  [^String app ^String env m]
  (mc/insert (events-bucket-name-for app env) m))

(def ^{:const true}
  default-limit 50)

(def ^{:const true}
  default-sort
  {:emitted_at -1})

(defn all-recent-events
  ([^String coll]
     (all-recent-events coll default-limit default-sort))
  ([^String coll ^long lmt srt]
     (with-collection coll
       (find {})
       (limit lmt)
       (sort srt))))

(defn all-tagged-events
  ([^String coll tags]
     (all-tagged-events coll tags default-limit default-sort))
  ([^String coll tags lmt srt]
     (with-collection coll
       (find {:tags {$in tags}})
       (limit lmt)
       (sort srt))
     (with-collection coll
       (find {:tags {$in tags}})
       (limit lmt)
       (sort srt))))

(defn all-events-on-host
  ([^String coll ^String hostname]
     (all-events-on-host coll hostname default-limit default-sort))
  ([^String coll ^String hostname lmt srt]
     (with-collection coll
       (find {:hostname hostname})
       (limit lmt)
       (sort srt))))

(defn- search-results-to-maps
  [results]
  (let [hits (:hits results)]
    (if (> (:total hits) 0)
      (map #(:_source %) (:hits hits))
      [])))

(defn most-recent-event-in
  [^String coll]
  (first (with-collection (str coll "_events")
           (fields [:received_at])
           (sort   {:received_at -1})
           (limit  1))))

;; TODO: these belong to Monger
(def ^{:const true}
  system-collection-pattern #"^(system|fs)")

(defn- system-collection?
  [^String coll-name]
  (re-find system-collection-pattern coll-name))

(defn- chop-off-suffix
  [^String s]
  (maybe-chopr s "_events"))

(defn all-app-names
  []
  (map chop-off-suffix (remove system-collection? (mdb/get-collection-names))))
