(ns clojurewerkz.eventoverse.collector.server
  (:gen-class)
  (:require [clojure.tools.cli :refer [cli]]
            [clojure.tools.logging :as log]
            [langohr.core          :as lh]
            [noir.server           :as web]
            [clojurewerkz.eventoverse.collector.amqp  :as amqp]
            [clojurewerkz.eventoverse.collector.udp   :as udp]
            [clojurewerkz.eventoverse.collector.store.mongo-event-store :as mongo-store])
  (:use aleph.http
        lamina.core))


;;
;; Implementation
;;

(def config {})

(defn- exit!
  "Logs an error and exits with non-zero code"
  [s]
  (log/error s)
  (System/exit 1))

(defn initialize-udp
  [options]
  (.start (Thread. ^Runnable (fn []
                               (when-let [udp-opts (:udp options)]
                                 (let [port     (get-in udp-opts [:connection :port])]
                                   (log/infof "Binding UDP listener to %s" port)
                                   (udp/connect (:connection udp-opts))
                                   (log/info "Connected to RabbitMQ")))))))

(defn initialize-amqp
  [options]
  (.start (Thread. ^Runnable (fn []
                               (when-let [amqp-opts (:amqp options)]
                                 (let [vhost     (get-in amqp-opts [:connection :vhost])
                                       username  (get-in amqp-opts [:connection :username])]
                                   (log/infof "Connecting to RabbitMQ as %s@%s" username vhost)
                                   (amqp/connect (:connection amqp-opts))
                                   (log/info "Connected to RabbitMQ")))))))


(defn- noir-mode-from
  [options]
  (let [env (keyword (get options :environment :development))]
    (case env
      :dev         :dev
      :development :dev
      :test        :test
      :staging     :production
      :production  :production)))

(defn initialize-custom
  [options]
  (if-let [initializer-config (get-in options [:custom-intializer])]
    (do
      (require (eval (:ns initializer-config)))
      (let [funk (ns-resolve (eval (:ns initializer-config)) (eval (:fn initializer-config)))]
        (funk  options)))))

(defn initialize-webserver
  [options]
  (.start (Thread. ^Runnable (fn []
                               (log/info "Loading Noir views")
                               (web/load-views-ns 'clojurewerkz.eventoverse.collector.web)
                               (when (get-in options [:custom-view-nss])
                                 (doseq [n (get-in options [:custom-view-nss])]
                                   (web/load-views-ns (eval n))))
                               (let [mode   (noir-mode-from options)
                                     port   (-> options :http_server :port)
                                     noir-f (web/gen-handler {:mode mode
                                                              :ns 'clojurewerkz.eventoverse.collector})]
                                 (log/infof "Starting Web server on port %d in %s mode" port mode)
                                 (start-http-server
                                  (wrap-ring-handler noir-f) {:port port :websocket true}))))))

(defn initialize-mongodb
  [options]
  (mongo-store/init-mongo-event-store (:mongodb options)))

(def initializers [initialize-mongodb
                   initialize-webserver
                   initialize-amqp
                   initialize-udp])



;;
;; API
;;

(defn start
  [options]
  (alter-var-root (var config) (constantly options))
  (dorun (map #(% options) initializers)))

(defn load-config-from-file
  [config-file]
  (read-string (slurp config-file)))

(defn -main
  [& args]
  (let [[options positional-args banner] (cli args
                                              ["--environment" "Environment to run in (development, production, et cetera)" :default "development"]
                                              ["--config-file" "Path to configuration file to use"])]
    (if-let [config (load-config-from-file (:config-file options))]
      (do
        (log/infof "Using config file at %s" (:config-file options))
        (start config))
      (exit! "Please either specify environment using --environment switch or config file path using --config-file switch"))
    nil))
