(ns clojurewerkz.eventoverse.collector.core
  (:require [clojurewerkz.eventoverse.collector.logging :as log]
            [clojurewerkz.eventoverse.collector.store   :as store])
  (:use [validateur.validation :only [validation-set presence-of] :as vld])
  (:import java.util.Date
           org.bson.types.ObjectId))

;;
;; Implementation
;;

(def ^{:doc "Internal (to Eventoverse) event consumers. This may be used to propagate events to
             various clients over WebSockets and the like, for internal event feed and so on."}
  internal-consumers (atom #{}))

(defn add-consumer
  [f]
  (swap! internal-consumers conj f))

(defn remove-consumer
  [f]
  (swap! internal-consumers disj f))

(defn broadcast
  [m ^String app ^String env]
  (doseq [f @internal-consumers]
    (f m app env)))



(def ^{ :const true :doc "Default event attributes, except for the timestamp" }
  default-event-attributes {:tags []})

(defn merge-defaults
  "Merges event attributes with default attributes"
  [event]
  (merge default-event-attributes {:received_at (Date.)} event))

(declare event-validator)
(defn- process-without-validation
  [{:keys [application environment] :as m}]
  (let [id (ObjectId.)
        m' (dissoc m :application :environment)]
    (store/insert-event application environment (merge m' {:_id id}))
    (broadcast (merge m {:_id id}) application environment)))


;;
;; API
;;

(def event-validator
  (validation-set
   (presence-of :type)
   (presence-of :application)
   (presence-of :environment)
   (presence-of :emitted_at)
   (presence-of :tags)))

(defn validate
  "Returns validation errors for the given event"
  [m]
  (event-validator m))

(defn valid?
  [m]
  (vld/valid? event-validator m))

(defn process
  [event]
  (let [m (merge-defaults event)]
    (if (valid? m)
      (process-without-validation m)
      (log/invalid-event (:application m) (:environment m) (event-validator m)))))

(defn custom-attributes
  "Returns custom (optional) attributes from the given event"
  [event]
  (dissoc event :type :application :environment :emitted_at :tags :custom-attributes))
