(ns clojurewerkz.eventoverse.collector.web.events
  (:require [noir.core :refer [defpage url-for]]
            [noir.response :as r]
            [stencil.core :as st]
            [clojurewerkz.eventoverse.collector.web.common :refer [in-layout
                                                                   format-tag format-tags tags-from
                                                                   format-emitted-at format-hostname
                                                                   inject-custom-attributes
                                                                   include-js
                                                                   custom-rows]]
            [clojurewerkz.eventoverse.collector.store :as store]))


;;
;; Implementation
;;


;;
;; API
;;

(defpage recent-events [:get "/apps/:app/env/:env/events/recent/"]
  {:keys [^String app ^String env]}
  (let [events (->> (store/all-recent-events (store/events-bucket-name-for app env))
                    (mapv #(format-tags % app env))
                    (mapv format-emitted-at)
                    (mapv #(format-hostname % app env))
                    (mapv inject-custom-attributes))
        event-rows  (custom-rows events "events/event_row")]
    (in-layout
     (st/render-file "templates/events/recent" {:event-rows event-rows
                                                :n (count events)
                                                :application app
                                                :environment env})
     {:javascripts (include-js "events/recent")})))

(defpage tagged-events [:get "/apps/:app/env/:env/events/tagged/:tagline"]
  {:keys [^String app ^String env ^String tagline]}
  (let [tags   (tags-from tagline)
        events (->> (store/all-tagged-events (store/events-bucket-name-for app env) tags)
                    (mapv #(format-tags % app env))
                    (mapv format-emitted-at)
                    (mapv #(format-hostname % app env))
                    (mapv inject-custom-attributes))]
    (in-layout
     (st/render-file "templates/events/tagged" {:events events
                                                :n (count events)
                                                :tags tags
                                                :tagline tagline
                                                :application app
                                                :environment env
                                                :app-path (url-for recent-events {:app app :env env})}))))


(defpage hostname-events [:get "/apps/:app/env/:env/events/on-host/:hostname"]
  {:keys [^String app ^String env ^String hostname]}
  (let [events (->> (store/all-events-on-host (store/events-bucket-name-for app env) hostname)
                    (mapv #(format-tags % app env))
                    (mapv format-emitted-at)
                    (mapv #(format-hostname % app env))
                    (mapv inject-custom-attributes))]
    (in-layout
     (st/render-file "templates/events/on_hostname" {:events events
                                                     :n (count events)
                                                     :hostname hostname
                                                     :application app
                                                     :environment env
                                                     :app-path (url-for recent-events {:app app :env env})}))))

(defpage event-search [:get "/apps/:app/env/:env/search/"]
  {:keys [^String app ^String env] :as params}
  (let [q      (:q params)
        events (->> (store/all-events-by-search (store/events-bucket-name-for app env) q)
                    (mapv #(format-tags % app env))
                    (mapv format-emitted-at)
                    (mapv #(format-hostname % app env))
                    (mapv inject-custom-attributes))
        event-rows  (custom-rows events "events/event_row")]
    (in-layout
     (st/render-file "templates/events/search" {:event-rows event-rows
                                                :n (count events)
                                                :q q
                                                :application app
                                                :environment env
                                                :app-path (url-for recent-events {:app app :env env})}))))